<?php

namespace App\Http\Controllers;

use App\Models\Clinic;
use App\Models\Doctor;
use App\Models\Speciality;
use Exception;
use App\Http\Resources\DoctorResource;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Models\ClinicDoctor;

class DoctorController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $doctors = Doctor::where('status', 'accepted')->get();

        return view('doctors.index', compact('doctors'));
    }


    public function getDoctors()
    {
        //
        try {
            $doctors = Doctor::where('status', 'accepted')->get();
            $data = DoctorResource::collection($doctors);

            return response()->json([
                'statusCode' => 200,
                'data' => $data
            ]);
        } catch (Exception $e) {
            return response()->json([
                'statusCode' => 500,
                'message' => $e->getMessage(),
            ]);
        }
    }

    public function doctorRequests()
    {
        return view('doctors.requests');
    }

    public function getPendingDoctors()
    {
        try {
            $doctors = Doctor::where('status', 'pending')->get();
            $data = DoctorResource::collection($doctors);

            return response()->json([
                'statusCode' => 200,
                'data' => $data
            ]);
        } catch (Exception $e) {
            return response()->json([
                'statusCode' => 500,
                'message' => $e->getMessage(),
            ]);
        }
    }



    public function create()
    {
        return view('doctors.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'name' => ['required', 'string'],
                'email' => ['nullable', 'email', 'unique:doctors,email'],
                'phone' => ['required', 'numeric', 'unique:doctors,phone'],
                'password' => ['nullable', 'string'],
                'gender' => ['nullable', 'string'],
                'job_title' => ['nullable', 'string'],
                'summary' => ['nullable', 'string'],
                'state' => ['nullable', 'string'],
                'city' => ['nullable', 'string'],
                'ragion' => ['nullable', 'string'],
                'specialty_id' => ['nullable', 'exists:specialities,id'],
                'secretary_number' => ['nullable', 'string'],
                'price' => ['nullable', 'numeric', 'min:0'],
                'image' => ['nullable', 'image'],
                'files' => ['nullable', 'file'],
            ]);

            $doctor = new Doctor();
            $doctor->name = $request->name;
            $doctor->phone = $request->phone;
            $doctor->email = $request->email;
            $doctor->gender = $request->gender;
            $doctor->job_title = $request->job_title;
            $doctor->summary = $request->summary;
            $doctor->state = $request->state;
            $doctor->city = $request->city;
            $doctor->ragion = $request->ragion;
            $doctor->specialty_id = $request->specialty_id;
            $doctor->confirmed_by = auth()->id();
            $doctor->secretary_number = $request->secretary_number;
            $doctor->price = $request->price;

            if ($request->password) {
                $doctor->password = Hash::make($request->password);
            }

            $doctor->save();

            if ($request->hasFile('image')) {
                $doctor->addMediaFromRequest('image')->toMediaCollection($doctor->phone);
            }

            if ($request->hasFile('files')) {
                try {
                    $doctor->addMediaFromRequest('files')->toMediaCollection($doctor->phone);
                } catch (\Exception $e) {
                    return redirect()->back()->with('error', 'Failed to upload files');
                }
            }

            return redirect()->route('doctors.index')->with('success', 'Doctor created successfully');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Doctor  $doctor
     * @return \Illuminate\Http\Response
     */
    public function show(Doctor $doctor)
    {
        // Eager load relationships to avoid N+1 queries
        $doctor->load(['clinics', 'specialty']);
        return view('doctors.show', compact('doctor'));
    }


    public function edit(Request $request, $id)
    {
        try {

            $doctor_id = $id;

            $doctor = Doctor::find($doctor_id);

            if (!$doctor) {
                // return response()->json(['error' => 'invalid doctor id'] , 404);
                return redirect()->back()->with('error', 'invalid doctor id');
            }

            $specialities = Speciality::get();
            $clinics = Clinic::get();
            $assignedClinics = ClinicDoctor::where('doctor_id', $id)->get();

            return view('doctors.edit', compact('doctor', 'specialities', 'clinics', 'assignedClinics'));
        } catch (Exception $ex) {
            return redirect()->back()->with('error', $ex->getMessage());
        }
    }

    public function updateDoctor(Request $request)
    {
        $doctor_id = $request->id;
        $doctor = Doctor::findorFail($doctor_id);

        try {
            // Validate the request
            $request->validate([
                'name' => ['required', 'string'],
                'email' => ['nullable', 'email', 'unique:doctors,email,' . $doctor_id],
                'phone' => ['required', 'numeric', 'unique:doctors,phone,' . $doctor_id],
                'gender' => ['nullable', 'string'],
                'job_title' => ['nullable', 'string'],
                'summary' => ['nullable', 'string'],
                'state' => ['nullable', 'string'],
                'city' => ['nullable', 'string'],
                'ragion' => ['nullable', 'string'],
                'secretary_number' => ['nullable', 'string'],
                'price' => ['nullable', 'numeric', 'min:0'],
                'speciality_id' => ['nullable', 'exists:specialities,id'],
                'avatar' => ['nullable', 'image', 'mimes:jpg,jpeg,png,gif'],
            ]);

            DB::beginTransaction();
            
            // Update basic doctor information
            $doctor->name = $request->name;
            $doctor->email = $request->email;
            $doctor->phone = $request->phone;
            $doctor->job_title = $request->job_title;
            $doctor->gender = $request->gender;
            $doctor->summary = $request->summary;
            $doctor->state = $request->state;
            $doctor->city = $request->city;
            $doctor->ragion = $request->ragion;
            $doctor->secretary_number = $request->secretary_number;
            $doctor->price = $request->price;
            $doctor->specialty_id  = $request->speciality_id ?? null;

            // Update clinic assignments
            if ($request->has('clinics') && is_array($request->clinics)) {

                ClinicDoctor::where('doctor_id', $doctor_id)->delete();

                foreach ($request->clinics as $clinicData) {
                    ClinicDoctor::create([
                        'doctor_id' => $doctor_id,
                        'clinic_id' => $clinicData['clinic_id'],
                        'am' => json_encode([
                            'from' => $clinicData['am_from'],
                            'to' => $clinicData['am_to'],
                        ]),
                        'pm' => json_encode([
                            'from' => $clinicData['pm_from'],
                            'to' => $clinicData['pm_to'],
                        ]),
                    ]);
                }
            }

            // Update profile image - use consistent collection name
            if ($request->hasFile('avatar') && $request->file('avatar')->isValid()) {
                // Clear existing media in the phone-based collection
                $doctor->clearMediaCollection($doctor->phone);
                // Add new image to the same collection used in registration
                $doctor->addMediaFromRequest('avatar')
                    ->toMediaCollection($doctor->phone);
            }

            $doctor->update();
            DB::commit();
            return redirect()->route('doctors.show', $request->id)->with('success', 'Doctor Updated Successfully');
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
    /**
     * Update the status resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Doctor  $doctor
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Doctor $doctor)
    {
        //
        $msg = '';
        if ($request->status == 'accept') {
            $doctor->status = 'accepted';
            $msg = 'Doctor Accepted successfully';
        }
        if ($request->status == 'reject') {
            $doctor->status = 'rejected';
            $msg = 'Doctor Rejected successfully';
        }
        $doctor->save();
        return redirect()->route('doctors.index')
            ->with('success', $msg);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Doctor  $doctor
     * @return \Illuminate\Http\Response
     */
    public function destroy(Doctor $doctor)
    {
        //
        $doctor->delete();

        return redirect()->route('doctors.index')
            ->with('success', 'doctor deleted successfully');
    }
}
