<?php

namespace App\Http\Controllers\Api;

use App\Models\User;
use App\Models\Doctor;
use App\Models\Patient;
use App\Models\File;
use App\Models\TermsAndCondition;
use Illuminate\Http\Request;
use App\Notifications\RegistsNotification;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Exception;
use JWTAuth;
use Validator;
use Mail;
use Notification;

class AuthController extends Controller
{
    // public function __construct()
    // {
    //     $this->middleware('auth:api', ['except' => ['login', 'register']]);
    // }

    public function login(Request $request)
    {
        $lang = $request->header('lang');

        $request->validate([
            'phone' => 'required',
            'password' => 'required',
        ]);

        if ($request->type == 'doctor') {
            // Check if account exists before attempting authentication
            $doctor = Doctor::where('phone', $request->phone)->first();
            
            if (!$doctor) {
                return response()->json([
                    'statusCode' => 404,
                    'message' => $lang == 'en' ? 'Account not found. Please register first.' : 'الحساب غير موجود. يرجى التسجيل أولاً',
                ]);
            }

            // Account exists, now attempt authentication
            $credentials = $request->only('phone', 'password');
            $token = auth('doctors')->attempt($credentials);

            if (!$token) {
                return response()->json([
                    'statusCode' => 401,
                    'message' => $lang == 'en' ? 'Invalid password. Please check your password and try again.' : 'كلمة المرور غير صحيحة. يرجى التحقق من كلمة المرور والمحاولة مرة أخرى',
                ]);
            }

            // Check account status
            if ($doctor->status != 'accepted') {
                return response()->json([
                    'statusCode' => 401,
                    'message' => $lang == 'en' ? 'your account not accepted yet' : 'لم يتم على الموافقه على حسابك الى الان',
                ]);
            }
            
            return response()->json([
                'data' => [
                    "id" => $doctor->id,
                    'type' => 'doctor',
                    "name" => $doctor->name,
                    "phone" => $doctor->phone,
                    "email" => $doctor->email,
                ],
                'token' => $token,
                'message' => $lang == 'en' ? "Successfully logged in"  : "تم تسجيل الدخول بنجاح",
                "statusCode" => 200,
            ]);
        }

        if ($request->type == 'patient') {
            // Check if account exists before attempting authentication
            $patient = Patient::where('phone', $request->phone)->first();
            
            if (!$patient) {
                return response()->json([
                    'statusCode' => 404,
                    'message' => $lang == 'en' ? 'Account not found. Please register first.' : 'الحساب غير موجود. يرجى التسجيل أولاً',
                ]);
            }

            // Account exists, now attempt authentication
            $credentials = $request->only('phone', 'password');
            $token = auth('patients')->attempt($credentials);

            if (!$token) {
                return response()->json([
                    'statusCode' => 401,
                    'message' => $lang == 'en' ? 'Invalid password. Please check your password and try again.' : 'كلمة المرور غير صحيحة. يرجى التحقق من كلمة المرور والمحاولة مرة أخرى',
                ]);
            }

            return response()->json([
                'data' => [
                    "id" => $patient->id,
                    'type' => 'patient',
                    "name" => $patient->name,
                    "phone" => $patient->phone,
                    "email" => $patient->email,
                ],
                'token' => $token,
                'message' => $lang == 'en' ? "Successfully logged in"  : "تم تسجيل الدخول بنجاح",
                "statusCode" => 200,
            ]);
        }
    }

    public function register(Request $request)
    {

        try {
            $lang = $request->header('lang');
            if ($request->type == 'doctor') {
                // Get active terms to validate against
                $activeTerms = TermsAndCondition::getActiveTerms();
                
                // Only require terms acceptance if active terms exist
                $validationRules = [
                    'name' => ['required', 'string'],
                    'email' => ['string', 'email', 'unique:doctors'],
                    'phone' => ['required', 'numeric', 'unique:doctors'],
                    'password' => ['required'],
                ];
                
                $customMessages = [];
                
                if ($activeTerms) {
                    $validationRules['terms_accepted'] = ['required', 'accepted'];
                    $customMessages = [
                        'terms_accepted.required' => $lang == 'en' ? 'You must accept the terms and conditions to register.' : 'يجب الموافقة على الشروط والأحكام للتسجيل.',
                        'terms_accepted.accepted' => $lang == 'en' ? 'You must accept the terms and conditions to register.' : 'يجب الموافقة على الشروط والأحكام للتسجيل.',
                    ];
                }
                
                $request->validate($validationRules, $customMessages);
                $doctor = new doctor();
                $doctor->name = $request->name;
                $doctor->phone = $request->phone;
                $doctor->email = $request->email;
                $doctor->gender = $request->gender;
                $doctor->job_title = $request->job_title;
                $doctor->summary = $request->summary;
                $doctor->state = $request->state;
                $doctor->city = $request->city;
                $doctor->ragion = $request->ragion;
                $doctor->password = Hash::make($request->password);
                $doctor->specialty_id = $request->specialty_id;
                $doctor->confirmed_by = $request->confirmed_by;
                $doctor->secretary_number = $request->secretary_number;

                // Save doctor first to get the ID
                $doctor->save();

                // Store terms acceptance
                if ($activeTerms) {
                    $doctor->terms_accepted_at = now();
                    $doctor->save();
                }

                // Upload profile image
                if ($request->hasFile('image')) {
                    try {
                        $doctor->addMediaFromRequest('image')
                            ->toMediaCollection($doctor->phone);
                    } catch (Exception $e) {
                        $doctor->delete();
                        return response()->json([
                            'status' => 422,
                            'message' => $lang == 'en' ? 'Failed to upload image: ' . $e->getMessage() : 'فشل فى تحميل الصورة: ' . $e->getMessage(),
                        ], 422);
                    }
                }

                // Upload certificate files (can be single file or array of files like files[0], files[1])
                // Laravel handles files[0] format automatically, so we check for 'files' key
                if ($request->hasFile('files')) {
                    try {
                        $files = $request->file('files');
                        
                        // If it's an array of files (files[0], files[1], etc.)
                        if (is_array($files)) {
                            foreach ($files as $index => $file) {
                                if ($file && $file->isValid()) {
                                    // For array files, we need to get the specific file from request
                                    $fileKey = "files.{$index}";
                                    if ($request->hasFile($fileKey)) {
                                        $doctor->addMediaFromRequest($fileKey)
                                            ->usingName($file->getClientOriginalName())
                                            ->usingFileName($file->getClientOriginalName())
                                            ->toMediaCollection($doctor->phone);
                                    } else {
                                        // Fallback: use the file object directly
                                        $doctor->addMedia($file->getRealPath())
                                            ->usingName($file->getClientOriginalName())
                                            ->usingFileName($file->getClientOriginalName())
                                            ->toMediaCollection($doctor->phone);
                                    }
                                }
                            }
                        } elseif ($files && $files->isValid()) {
                            // Single file
                            $doctor->addMediaFromRequest('files')
                                ->usingName($files->getClientOriginalName())
                                ->usingFileName($files->getClientOriginalName())
                                ->toMediaCollection($doctor->phone);
                        }
                    } catch (Exception $e) {
                        // If file upload fails, delete the doctor and return error
                        $doctor->delete();
                        return response()->json([
                            'status' => 422,
                            'message' => $lang == 'en' ? 'Failed to upload files: ' . $e->getMessage() : 'فشل فى تحميل الملفات: ' . $e->getMessage(),
                        ], 422);
                    }
                }
                $credentials = $request->only('phone', 'password');
                $token = auth('doctors')->attempt($credentials);

                return response()->json([
                    'status' => 200,
                    'data' => [
                        'id' => $doctor->id,
                        'type' => 'doctor',
                        'name' => $doctor->name,
                        'phone' => $doctor->phone,
                        'email' => $doctor->email,
                        'access_token' => $token
                    ],
                    'message' => $lang == 'en' ? "Account Created Succesfully"  : "تم انشاء الحساب",
                ],200);
            }

            if ($request->type == 'patient') {
                $request->validate([
                    'name' => ['required', 'string'],
                    'email' => ['string', 'email', 'unique:patients'],
                    'phone' => ['required', 'numeric', 'unique:patients'],
                    'password' => ['required'],
                ]);

                $patient = new Patient();
                $patient->name = $request->name;
                $patient->phone = $request->phone;
                $patient->email = $request->email;
                $patient->password = Hash::make($request->password);
                $patient->gender = $request->gender;
                $patient->birthdate = $request->birthdate;
                $patient->address = $request->address;

                if ($request->hasFile('image')) {
                    $patient->addMediaFromRequest('image')
                        ->toMediaCollection($patient->phone);
                }

                $patient->save();

                $credentials = $request->only('phone', 'password');
                $token = auth('patients')->attempt($credentials);
                return response()->json([
                    'statusCode' => 200,
                    'data' => [
                        'id' => $patient->id,
                        'type' => 'patient',
                        'name' => $patient->name,
                        'phone' => $patient->phone,
                        'email' => $patient->email,
                        'access_token' => $token
                    ],
                    'message' => $lang == 'en' ? "Account Created Succesfully"  : "تم انشاء الحساب بنجاح",
                ],200 );
            }
        } catch (Exception $e) {
            return response()->json([
                'statusCode' => 435,
                'message' => $e->getMessage(),
            ],435);
        }
    }

    public function logout(Request $request)
    {
        $lang = $request->header('lang');

        Auth::logout();
        return response()->json([
            'statusCode' => 200,
            'message' => $lang == 'en' ? 'Successfully logged out' : 'تم تسجيل الخروج بنجاح',
        ]);
    }
}
