<?php

namespace App\Services;

use App\Models\User;
use App\Models\Role;
use App\Models\Permission;
use App\Models\Team;
use App\Models\Country;
use App\Models\PhoneNumber;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Password;

class UserService
{
    /**
     * Get users based on current user's access level
     */
    public function getUsersForCurrentUser($currentUser)
    {
        if ($currentUser->type == 'admin' || $currentUser->role->name == 'Operation Specialist') {
            return User::where('is_blocked', 0)->get();
        } else {
            $users = User::where('is_blocked', 0)->get();
            $userChildren = array();
            
            foreach ($users as $user) {
                $parents = $user->getParents()->pluck('id')->toArray();
                if (in_array($currentUser->id, $parents)) {
                    array_push($userChildren, [
                        'id' => $user->id,
                        'name' => $user->name
                    ]);
                }
            }
            
            return $userChildren;
        }
    }

    /**
     * Get user permissions
     */
    public function getUserPermissions(User $user)
    {
        $userPermissions = $user->permissions()->pluck('name');
        return $userPermissions->isEmpty() ? '' : $userPermissions;
    }

    /**
     * Get user children for editing
     */
    public function getUserChildrenForEdit($currentUser)
    {
        if ($currentUser->type == 'admin') {
            return User::get();
        } else {
            $users = User::get();
            $userChildren = array();
            
            foreach ($users as $user) {
                $parents = $user->getParents()->pluck('id')->toArray();
                if (in_array($currentUser->id, $parents)) {
                    array_push($userChildren, [
                        'id' => $user->id,
                        'name' => $user->name
                    ]);
                }
            }
            
            return $userChildren;
        }
    }

    /**
     * Update user basic information
     */
    public function updateUserBasicInfo(User $user, Request $request)
    {
        if ($request->password) {
            $user->password = Hash::make($request->password);
            $user->save();
        }
        
        if ($request->role_id) {
            $user->update([
                'name' => $request->name,
                'email' => $request->email,
                'role_id' => $request->role_id,
                'assigned_to' => $request->assigned_to
            ]);
        } else {
            $user->update([
                'name' => $request->name,
                'email' => $request->email,
            ]);
        }
        
        return $user;
    }

    /**
     * Update user phone numbers
     */
    public function updateUserPhoneNumbers(User $user, Request $request)
    {
        // Delete existing phone numbers
        if (count($user->phones) != 0) {
            foreach ($user->phones as $phone) {
                $phone->delete();
            }
        }

        // Add new phone numbers
        if ($request->mobile_number) {
            $counts = count($request->mobile_number);
            for ($i = 0; $i < $counts; $i++) {
                $phones = new PhoneNumber;
                $phones->phone = $request->mobile_number[$i];
                $phones->user_id = $user->id;
                $phones->country_id = $request->country_id[$i];
                $phones->save();
            }
        }
    }

    /**
     * Update user permissions
     */
    public function updateUserPermissions(User $user, Request $request)
    {
        if ($request->permissions) {
            $user->syncPermissions($request->permissions);
        } else {
            $user->detachPermissions($user->permissions);
        }
    }

    /**
     * Get user type for profile
     */
    public function getUserType($userId)
    {
        $role_id = DB::table('role_user')->where('user_id', $userId)->first();

        if ($role_id) {
            $userType = DB::table('roles')->where('id', $role_id->role_id)->first();
            return $userType->name;
        } else {
            return null;
        }
    }

    /**
     * Update user profile
     */
    public function updateUserProfile(User $user, Request $request)
    {
        if ($request->password != NULL) {
            $user->update([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password)
            ]);
        } else {
            $user->update([
                'name' => $request->name,
                'email' => $request->email
            ]);
        }
        
        return $user;
    }

    /**
     * Block user
     */
    public function blockUser($userId)
    {
        $user = User::find($userId);
        $user->is_blocked = 1;
        $user->save();
        return $user;
    }

    /**
     * Unblock user
     */
    public function unblockUser($userId)
    {
        $user = User::find($userId);
        $user->is_blocked = 0;
        $user->save();
        return $user;
    }

    /**
     * Get blocked users
     */
    public function getBlockedUsers()
    {
        return User::where('is_blocked', 1)->get();
    }

    /**
     * Get users tree data
     */
    public function getUsersTreeData()
    {
        $allUsers = User::get();
        $users = User::where('assigned_to', null)->get();
        $parentCodes = User::where('assigned_to', '!=', null)->get();
        $arrayOfCodes = [];
        
        foreach ($parentCodes as $code) {
            array_push($arrayOfCodes, $code->assigned_to);
        }
        
        return [
            'users' => $users,
            'arrayOfCodes' => $arrayOfCodes,
            'allUsers' => $allUsers
        ];
    }
} 