<?php

namespace App\Services;

use App\Models\Communication;
use App\Models\User;
use App\Models\Lead;
use App\Models\CallAction;
use App\Models\Event;
use App\Models\Broker;
use App\Models\Agent;
use App\Models\Status;
use App\Models\CommunicationEvent;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\Builder;
use Carbon\Carbon;

class CommunicationService
{
    public function logActivity(string $message): void
    {
        activity()
            ->event('visit')
            ->log(auth()->user()->name . $message);
    }

    public function getFilterData(): array
    {
        return [
            'creators' => User::get(),
            'callActions' => CallAction::get(),
            'leads' => Lead::get(),
            'events' => Event::get(),
        ];
    }

    public function applyFilters(Builder $communications, Request $request, &$filterData): void
    {
        if ($request->creators_id) {
            $communications->whereIn('user_id', $request->creators_id);
        }

        if ($request->leads_id) {
            $communications->whereIn('lead_id', $request->leads_id);
        }

        if ($request->callActions_id) {
            $communications->whereIn('call_action_id', $request->callActions_id);
        }

        if ($request->events_id) {
            $filterData['events'] = $filterData['events']->whereIn('id', $request->events_id);
        }

        if ($request->time1 && $request->time2) {
            $communications->whereTime('created_at', '>=', $request->time1)
                ->whereTime('created_at', '<=', $request->time2);
        }

        if ($request->date1 && $request->date2) {
            $communications->whereDate('created_at', '>=', $request->date1)
                ->whereDate('created_at', '<=', $request->date2);
        }
    }

    public function applyUserAccessControl(Builder $communications, $creators): void
    {
        $currentUser = auth()->user();
        $userRole = $currentUser->role->name;
        
        if ($userRole != 'Sales Admin' && $userRole != 'Sales Director') {
            if ($currentUser->type != 'admin') {
                $userChildren = $this->getUserChildren($currentUser, $creators);
                $communications->whereIn('user_id', $userChildren);
            }
        }
    }

    private function getUserChildren($currentUser, $creators): array
    {
        $userChildren = [$currentUser->id];
        
        foreach ($creators as $user) {
            $parents = $user->getParents()->pluck('id')->toArray();
            if (in_array($currentUser->id, $parents)) {
                $userChildren[] = $user->id;
            }
        }
        
        return $userChildren;
    }

    public function createBrokerCommunication(array $data): Communication
    {
        $broker = Broker::find($data['broker_id']);
        
        $communication = new Communication();
        $communication->type = $data['type'];
        $communication->state = $data['state'];
        $communication->broker_id = $data['broker_id'];
        $communication->agent_id = $data['agent_id'];
        $communication->reminder = $data['reminder'] ?? '';
        $communication->description = $data['description'];
        $communication->check_in = $data['check_in'] ?? 0;
        $communication->user_id = auth()->user()->id;
        
        if (isset($data['reminder'])) {
            $reminder = Carbon::create($data['reminder']);
            $communication->reminder = $reminder;
        }
        
        $communication->save();
        
        return $communication;
    }

    public function createBrokerEvent(Communication $communication, array $data): void
    {
        if (isset($data['start_date'])) {
            $event = new Event();
            $event->title = 'make ' . $communication->type . ' With ' . $communication->broker?->name;
            $event->type = $communication->type;
            $event->communication_id = $communication->id;
            
            $start_date = Carbon::create($data['start_date']);
            $event->start = $start_date;

            if (isset($data['end_date'])) {
                $end_date = Carbon::create($data['end_date']);
                $event->end = $end_date;
            }

            $event->save();
            
            CommunicationEvent::create([
                'communication_id' => $communication->id,
                'event_id' => $event->id
            ]);
        }
    }

    public function createLeadCommunication(array $data): Communication
    {
        $lead = Lead::find($data['lead_id']);
        $callAction = CallAction::find($data['call_action_id']);
        $status = Status::where('name', $callAction->name)->first();
        
        $communication = new Communication();
        $communication->call_action_id = $data['call_action_id'];
        $communication->action_method = $data['action_method'];
        $communication->status_id = $status->id;
        $communication->state = $data['start_date'] ? $data['state'] : 'done';
        $communication->description = $data['description'];
        $communication->lead_id = $data['lead_id'];
        $communication->user_id = $data['user_id'] ?? auth()->user()->id;
        $communication->call_action_id = $data['call_action_id'];
        $communication->created_by = auth()->user()->id;

        $communication->save();

        // Update lead status
        $lead->status_id = $status->id;
        $lead->save();
        
        return $communication;
    }

    public function createLeadEvent(Communication $communication, array $data): void
    {
        if (isset($data['start_date'])) {
            $start_date = Carbon::create($data['start_date']);
            
            $event = new Event();
            $event->title = $communication->lead->name . $communication->user->name;
            $event->start = $data['start_date'];
            $event->type = 'call';
            $event->communication_id = $communication->id;
            $event->end = $data['start_date'];
            $event->save();
            
            CommunicationEvent::create([
                'communication_id' => $communication->id,
                'event_id' => $event->id
            ]);
        }
    }

    public function logBrokerActivity(Communication $communication, Broker $broker): void
    {
        $currentUser = auth()->user();
        activity()
            ->performedOn($communication)
            ->causedBy($currentUser)
            ->event('Activity')
            ->withProperties([
                'action_creator_name' => $currentUser->name,
                'action_creator_id' => $currentUser->id,
                'action_type' => $communication->type,
                'broker_name' => $communication->broker->name,
                'broker_id' => $communication->broker->id,
                'action_description' => $communication->description,
            ])
            ->log($currentUser->name . 'make and activity with: (' . $broker->name . ')');
    }

    public function logLeadActivity(Communication $communication, Lead $lead): void
    {
        $currentUser = auth()->user();
        activity()
            ->performedOn($communication)
            ->causedBy($currentUser)
            ->event('Call Action')
            ->withProperties([
                'action_creator_name' => $currentUser->name,
                'action_creator_id' => $currentUser->id,
                'action_type' => 'Call Action',
                'lead_name' => $lead->name,
                'lead_id' => $lead->id,
                'lead_status' => $communication->CommunicationStatus->name,
                'action_description' => $communication->description,
            ])
            ->log($currentUser->name . 'Took an action on lead: (' . $lead->name . ')');
    }

    public function getCommunicationShowData(int $communicationId): array
    {
        $communication = Communication::findOrFail($communicationId);
        
        if (isset($communication->lead)) {
            $lead = Lead::find($communication->lead_id);
        } else {
            $lead = Lead::withTrashed()->where('id', $communication->lead_id)->first();
        }
        
        if (isset($communication->user)) {
            $user = User::find($communication->user_id);
        } else {
            $user = User::withTrashed()->where('id', $communication->user_id)->first();
        }
        
        $users = User::get();
        $broker = Broker::find($communication->broker_id);
        
        if ($broker != null) {
            $agents = $broker->agents;
        } else {
            $agents = array();
        }
        
        return compact('communication', 'lead', 'user', 'users', 'broker', 'agents');
    }

    public function getUserChildrenForNextActions(): array
    {
        $currentUser = auth()->user();
        $users = User::get();

        $children = [$currentUser->id];
        foreach ($users as $user) {
            $parents = $user->getParents()->pluck('id')->toArray();

            if (in_array($currentUser->id, $parents)) {
                $children[] = $user->id;
            }
        }

        return $children;
    }

    public function updateCommunication(Communication $communication, array $data): void
    {
        $communication->type = $data['type'];
        $communication->state = $data['state'];
        $communication->broker_id = $data['broker_id'];
        $communication->agent_id = $data['agent_id'];
        $communication->reminder = $data['reminder'] ?? '';
        $communication->description = $data['description'];
        
        if (isset($data['reminder'])) {
            $reminder = Carbon::create($data['reminder']);
            $communication->reminder = $reminder;
        }
        
        $communication->save();
    }

    public function updateCommunicationEvent(Communication $communication, array $data): void
    {
        $event = $communication->event;
        $event->title = 'make ' . $communication->type . ' With ' . $communication->broker?->name;
        $event->type = $communication->type;
        $event->communication_id = $communication->id;
        
        $start_date = Carbon::create($data['start_date']);
        $event->start = $start_date;

        $event->save();
    }

    public function logCommunicationUpdate(Communication $communication): void
    {
        $currentUser = auth()->user();
        activity()
            ->performedOn($communication)
            ->causedBy($currentUser)
            ->event('Activity')
            ->withProperties([
                'action_creator_name' => $currentUser->name,
                'action_creator_id' => $currentUser->id,
                'action_type' => $communication->type,
                'broker_name' => $communication->broker?->name,
                'broker_id' => $communication->broker?->id,
                'action_description' => $communication->description,
            ])
            ->log($currentUser->name . 'make and activity with: (' . $communication->broker?->name . ')');
    }

    public function checkInCommunication(Communication $communication, array $data): void
    {
        $communication->check_in = 1;
        $communication->description = $data['description'];
        $communication->user_id = auth()->user()->id;
        $communication->save();

        $event = Event::where('communication_id', $data['communication_id'])->first();
        
        if (isset($data['start_date'])) {
            $start_date = Carbon::create($data['start_date']);
            $event->start = $start_date;
        }
        
        if (isset($data['end_date'])) {
            $end_date = Carbon::create($data['end_date']);
            $event->end = $end_date;
        }

        $event->save();
    }

    public function logCheckInActivity(Communication $communication): void
    {
        $currentUser = auth()->user();
        activity()
            ->performedOn($communication)
            ->causedBy($currentUser)
            ->event('Activity Check In')
            ->withProperties([
                'activity_creator_name' => $currentUser->name,
                'activity_creator_id' => $currentUser->id,
                'activity_type' => 'Activity Check In',
                'broker_name' => $communication->broker->name,
                'broker_id' => $communication->broker->id,
                'activity_type' => $communication->type,
                'activity_description' => $communication->description,
            ])
            ->log($currentUser->name . 'created an activity on lead: (' . $communication->broker->name . ')');
    }

    public function reassignCommunication(Communication $communication, int $userId): void
    {
        $communication->user_id = $userId;
        $communication->save();
    }

    public function logReassignActivity(Communication $communication): void
    {
        $currentUser = auth()->user();
        activity()
            ->performedOn($communication)
            ->causedBy($currentUser)
            ->event('Activity Check In')
            ->withProperties([
                'activity_creator_name' => $currentUser->name,
                'activity_creator_id' => $currentUser->id,
                'activity_type' => 're-assign user',
                'broker_name' => $communication->broker->name,
                'broker_id' => $communication->broker->id,
                'communication_id' => $communication->id,
            ])
            ->log($currentUser->name . ' Re-Assigned Activity to : (' . $communication->user->name . ')');
    }
} 