<?php

namespace App\Services;

use Auth;
use Carbon\Carbon;
use App\Models\File;
use App\Models\Lead;
use App\Models\Unit;
use App\Models\User;
use App\Models\Batch;
use App\Models\Offer;
use App\Models\Client;
use App\Models\Source;
use App\Models\Stream;
use App\Models\Country;
use App\Models\Gallery;
use App\Models\Payment;
use App\Models\Project;
use App\Models\Campaign;
use App\Models\Facility;
use App\Models\PhoneNumber;
use App\Models\FacilityUnit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Notification;
use Spatie\Activitylog\Models\Activity;

class ClientService
{
    /**
     * Get filtered clients based on request parameters
     */
    public function getFilteredClients(Request $request, $user)
    {
        $clients = Client::query();
        $userClients = $user->accessed_clients->pluck('id')->toArray();

        // Apply user access restrictions
        if ($user->type != 'admin' && !$user->role->hasPermission('clients-readAll')) {
            $clients->whereIn('id', $userClients);
        }

        // Apply filters
        $this->applyNameFilter($clients, $request);
        $this->applyCountryFilter($clients, $request);
        $this->applyPhoneFilter($clients, $request);
        $this->applyTimeFilter($clients, $request);
        $this->applyDateFilter($clients, $request);
        $this->applyCampaignFilter($clients, $request);
        $this->applySourceFilter($clients, $request);

        return $clients;
    }

    /**
     * Apply name filter to clients query
     */
    private function applyNameFilter($clients, Request $request)
    {
        if ($request->name) {
            $clients->where('name', 'LIKE', '%' . $request->name . '%');
        }
    }

    /**
     * Apply country filter to clients query
     */
    private function applyCountryFilter($clients, Request $request)
    {
        if ($request->country_id) {
            $clients_id = PhoneNumber::where('country_id', $request->country_id)
                ->where('client_id', '!=', NULL)
                ->pluck('client_id')
                ->toArray();
            $clients->whereIn('id', $clients_id);
        }
    }

    /**
     * Apply phone filter to clients query
     */
    private function applyPhoneFilter($clients, Request $request)
    {
        if ($request->phone) {
            $clients_id = PhoneNumber::where('phone', 'LIKE', '%' . $request->phone . '%')
                ->where('client_id', '!=', NULL)
                ->pluck('client_id')
                ->toArray();
            $clients->whereIn('id', $clients_id);
        }
    }

    /**
     * Apply time filter to clients query
     */
    private function applyTimeFilter($clients, Request $request)
    {
        if ($request->time1 && $request->time2) {
            $clients->whereTime('created_at', '>=', $request->time1)
                ->whereTime('created_at', '<=', $request->time2);
        }
    }

    /**
     * Apply date filter to clients query
     */
    private function applyDateFilter($clients, Request $request)
    {
        if ($request->date1 && $request->date2) {
            $clients->whereDate('created_at', '>=', $request->date1)
                ->whereDate('created_at', '<=', $request->date2);
        }
    }

    /**
     * Apply campaign filter to clients query
     */
    private function applyCampaignFilter($clients, Request $request)
    {
        if ($request->campaign_id) {
            $clients->whereIn('campaign_id', $request->campaign_id);
        }
    }

    /**
     * Apply source filter to clients query
     */
    private function applySourceFilter($clients, Request $request)
    {
        if ($request->source_id) {
            $clients->whereIn('source_id', $request->source_id);
        }
    }

    /**
     * Get maximum number of phone numbers for clients
     */
    public function getMaxPhoneNumbers($clients)
    {
        $clientNumbers = 0;
        foreach ($clients as $client) {
            $maxCount = count($client->phone_numbers);
            if ($maxCount > $clientNumbers) {
                $clientNumbers = $maxCount;
            }
        }
        return $clientNumbers;
    }

    /**
     * Create a new client
     */
    public function createClient(Request $request)
    {
        $client = new Client();
        $client->lead_id = $request->leadId;
        $client->name = $request->name;
        $client->email = $request->email;
        $client->job_title = $request->job_title;
        $client->came_from = $request->came_from;
        $client->user_id = $request->user_id ? $request->user_id : Auth()->user()->id;
        $client->campaign_id = $request->campaign_id;
        $client->source_id = $request->source_id;
        $client->civil_date = $request->civil_date ?? '';
        $client->civil_city = $request->civil_city ?? '';
        $client->civil_number = $request->civil_number ?? '';
        $client->civil_address = $request->civil_address ?? '';
        $client->nationality = $request->nationality ?? '';
        $client->save();

        return $client;
    }

    /**
     * Update lead when client is created
     */
    public function updateLeadForClient($leadId, $clientId)
    {
        $lead = Lead::find($leadId);
        $lead->is_client = 1;
        $lead->client_id = $clientId;
        $lead->save();
    }

    /**
     * Update phone numbers for client
     */
    public function updatePhoneNumbersForClient($mobileNumbers, $clientId)
    {
        if ($mobileNumbers) {
            for ($i = 0; $i < count($mobileNumbers); $i++) {
                $phone = PhoneNumber::where('phone', $mobileNumbers[$i])->first();
                if (isset($phone)) {
                    $phone->client_id = $clientId;
                    $phone->save();
                }
            }
        }
    }

    /**
     * Update client information
     */
    public function updateClient(Client $client, Request $request)
    {
        $client->name = $request->name;
        $client->email = $request->email;
        $client->job_title = $request->job_title;
        $client->came_from = $request->came_from;
        $client->civil_date = $request->civil_date;
        $client->civil_city = $request->civil_city;
        $client->civil_number = $request->civil_number;
        $client->civil_address = $request->civil_address;
        $client->nationality = $request->nationality ?? '';

        if ($request->source_id) {
            $source = Source::find($request->source_id);
            $campaign = $source->campaign;
            $client->source_id = $source->id;
            $client->campaign_id = $campaign->id;
        }

        $client->save();
        return $client;
    }

    /**
     * Update lead when client is updated
     */
    public function updateLeadForUpdatedClient(Client $client)
    {
        if ($client->lead_id != NULL) {
            $lead = Lead::find($client->lead_id);
            $lead->name = $client->name;
            $lead->save();
        }
    }

    /**
     * Get reservation form data
     */
    public function getReservationFormData($clientId, $unitId = null)
    {
        $client = Client::find($clientId);
        $projects = Project::where('is_active', '1')->get();
        $unit = '';
        $payment = '';
        $facilities = array();

        if ($unitId) {
            $unit = Unit::find($unitId);
            $payment = Payment::where('unit_id', $unitId)->first();
            $u_facilities = FacilityUnit::where('unit_id', $unitId)->get();
            foreach ($u_facilities as $facility) {
                $f = Facility::find($facility->facility_id);
                array_push($facilities, [
                    'name' => $f->name,
                    'price' => $facility->price,
                    'price_word' => $facility->price_word,
                ]);
            }
        }

        return compact('client', 'projects', 'unit', 'facilities', 'payment');
    }

    /**
     * Get units for project
     */
    public function getUnitsForProject($projectId)
    {
        $project = Project::find($projectId);
        $units = Unit::whereHas('building.phase.project', function ($query) use ($project) {
            $query->where('id', $project->id);
        })->where(['status' => 'free', 'is_active' => 1])->get();
        $facilities = $project->facilities;
        
        return ['units' => $units, 'facilities' => $facilities];
    }

    /**
     * Assign unit to client
     */
    public function assignUnit(Request $request)
    {
        $unit = Unit::find($request->unit_id);
        $client = Client::find($request->client_id);
        $project = Project::find($request->project_id);

        $unit->status = $request->status ?? 'hold';
        $unit->client_id = $client->id;
        $unit->Booked_date = Carbon::now()->format('y-m-d');

        if (isset($client->lead)) {
            $unit->source_id = $client->lead->source_id ?? null;
        }

        $unit->save();

        $this->updateFacilitiesForUnit($request);
        $this->createPaymentForUnit($request, $unit, $client, $project);
        $this->updateBatchesForUnit($unit, $client);

        return $unit;
    }

    /**
     * Update facilities for unit
     */
    private function updateFacilitiesForUnit(Request $request)
    {
        if (isset($request->facility_id)) {
            $oldFacilities = FacilityUnit::where('unit_id', $request->unit_id)->get();
            if (count($oldFacilities) != 0) {
                foreach ($oldFacilities as $oldFacility) {
                    $oldFacility->delete();
                }
            }
            
            for ($s = 0; $s < count($request->facility_id); $s++) {
                $facility_units = new FacilityUnit();
                $facility_units->unit_id = $request->unit_id;
                $facility_units->facility_id = $request->facility_id[$s];
                $facility_units->price = $request->facility_price[$s];
                $facility_units->price_word = $request->facility_price_word[$s];
                $facility_units->save();
            }
        }
    }

    /**
     * Create payment for unit
     */
    private function createPaymentForUnit(Request $request, $unit, $client, $project)
    {
        $payment = Payment::firstOrCreate(['unit_id' => $unit->id]);
        $payment->client_id = $client->id;
        $payment->unit_id = $unit->id;
        $payment->delivery_date = $request->delivery_date;

        // Facilities price
        $facilities_price = FacilityUnit::where('unit_id', $unit->id)->sum('price');

        // Total Price
        $total_price = $unit->unit_total_price + $facilities_price;

        // Maintenance
        $mentainance_value = $project->mentainance / 100;
        $mentainance = $total_price * $mentainance_value;

        // Total Amount
        $total_amount = $total_price + $mentainance;

        // Remaining
        $remaining = $total_amount - $request->reservation + $request->down_payment;

        // Storing Data
        $payment->mentainance = $mentainance;
        $payment->total_price = $total_price;
        $payment->total_amount = $total_amount;
        $payment->reservation = $request->reservation;
        $payment->down_payment = $request->down_payment + $request->reservation;
        $payment->remaining = $remaining;
        $payment->save();
    }

    /**
     * Update batches for unit
     */
    private function updateBatchesForUnit($unit, $client)
    {
        $project = $unit->building->phase->project;
        $facilities_price = FacilityUnit::where('unit_id', $unit->id)->sum('price');
        $total_price = $unit->unit_total_price + $facilities_price;
        $mentainance_value = $project->mentainance / 100;
        $mentainance = $total_price * $mentainance_value;

        foreach ($unit->batches as $batch) {
            $percentage = $batch->amount / 100;

            if ($batch->type == 'maintenance') {
                $amount = $mentainance * $percentage;
            } else {
                $amount = $total_price * $percentage;
            }

            $batch->amount = $amount;
            $batch->client_id = $client->id;
            $batch->save();
        }
    }

    /**
     * Handle file upload for client
     */
    public function handleFileUpload(Request $request)
    {
        $client = Client::find($request->client_id);

        if (isset($request['files'])) {
            return $this->uploadFiles($request, $client);
        }

        if (isset($request['images'])) {
            return $this->uploadImages($request, $client);
        }

        return redirect()->back()->with('error', 'No files selected');
    }

    /**
     * Upload files for client
     */
    private function uploadFiles(Request $request, $client)
    {
        for ($i = 0; $i < count($request['files']); $i++) {
            $fileName = time() . $request['files'][$i]->getClientOriginalName();
            $extension = $request['files'][$i]->extension();

            if ($extension == 'csv' || $extension == 'txt' || $extension == 'xlsx' || $extension == 'pdf' || $extension == 'docx') {
                $request['files'][$i]->move(public_path('files'), $fileName);

                $file = new File();
                $file->client_id = $request->client_id;
                $file->file = $fileName;
                $file->save();
            } else {
                return redirect()->back()->with('error', 'File type not allowed');
            }
        }
        return redirect()->route('clients.show', $client->id)->with('success', 'Files uploaded successfully');
    }

    /**
     * Upload images for client
     */
    private function uploadImages(Request $request, $client)
    {
        for ($i = 0; $i < count($request['images']); $i++) {
            $imageName = time() . $request['images'][$i]->getClientOriginalName();
            $extension = $request['images'][$i]->extension();

            if ($extension == 'png' || $extension == 'jpg' || $extension == 'jpeg') {
                $request['images'][$i]->move(public_path('images'), $imageName);

                $image = new Gallery();
                $image->client_id = $request->client_id;
                $image->name = $imageName;
                $image->save();
            } else {
                return redirect()->back()->with('error', 'Image type not allowed');
            }
        }
        return redirect()->route('clients.show', $client->id)->with('success', 'Images uploaded successfully');
    }

    /**
     * Get reservation contract data
     */
    public function getReservationContractData($unitId, $clientId)
    {
        $unit = Unit::find($unitId);
        $client = Client::find($clientId);
        $payment = Payment::where(['unit_id' => $unit->id, 'client_id' => $client->id])->first();

        $c_date = Carbon::create($unit->Booked_date);
        $c_date_formate = $c_date->isoFormat('dddd');
        $c_day = $this->getArabicDay($c_date_formate);

        return compact('unit', 'client', 'payment', 'c_day');
    }

    /**
     * Get Arabic day name
     */
    private function getArabicDay($englishDay)
    {
        $days = [
            'Saturday' => 'السبت',
            'Sunday' => 'الاحد',
            'Monday' => 'الاثنين',
            'Tuesday' => 'الثلاثاء',
            'Wednesday' => 'الاربعاء',
            'Thursday' => 'الخميس',
            'Friday' => 'الجمعه',
        ];

        return $days[$englishDay] ?? '';
    }

    /**
     * Store reservation contract
     */
    public function storeReservationContract(Request $request)
    {
        $unit = Unit::find($request->unit_id);
        $client = Client::find($request->client_id);
        $payment = Payment::where(['unit_id' => $unit->id, 'client_id' => $client->id])->first();

        $unit->Booked_date = $request->booked_date;
        $unit->save();

        $c_date = Carbon::create($unit->Booked_date);
        $c_date_formate = $c_date->isoFormat('dddd');
        $c_day = $this->getArabicDay($c_date_formate);

        $payment->batches_amount = $request->batches_amount;
        $payment->payment_method = $request->payment_method;
        $payment->batches_qty = $request->batches_qty;
        $payment->batches_payment_duration = $request->batches_payment_duration;
        $payment->save();

        return compact('unit', 'client', 'payment', 'c_day');
    }

    /**
     * Convert client to lead
     */
    public function convertClientToLead($clientId)
    {
        $client = Client::find($clientId);
        $lead = new Lead();
        $lead->name = $client->name;
        $lead->email = $client->email;
        $lead->job_title = $client->job_title;
        $lead->is_client = 1;
        $lead->client_id = $clientId;
        $lead->created_by = auth()->user()->id;
        $lead->campaign_id = $client->campaign_id;
        $lead->source_id = $client->source_id;
        $lead->save();

        $client->lead_id = $lead->id;
        $client->save();

        foreach ($client->phone_numbers as $phone) {
            $phone->lead_id = $lead->id;
            $phone->client_id = $client->id;
            $phone->save();
        }

        return $lead;
    }

    /**
     * Get archived clients
     */
    public function getArchivedClients()
    {
        return Client::onlyTrashed()->get();
    }

    /**
     * Restore client
     */
    public function restoreClient($clientId)
    {
        Client::withTrashed()->find($clientId)->restore();
        return Client::find($clientId);
    }

    /**
     * Restore all clients
     */
    public function restoreAllClients()
    {
        return Client::onlyTrashed()->restore();
    }

    /**
     * Delete image
     */
    public function deleteImage($imageId)
    {
        return Gallery::find($imageId)->delete();
    }

    /**
     * Update client access users
     */
    public function updateClientAccessUsers($clientId, $userIds)
    {
        $client = Client::find($clientId);
        $client->accessed_users()->sync($userIds);
        return $client;
    }
} 