<?php

namespace App\Services;

use App\Models\Lead;
use App\Models\User;
use App\Models\Status;
use App\Models\Campaign;
use App\Models\Country;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\Builder;

class CilService
{
    public function logCILsActivity(): void
    {
        $model = new Lead();
        $currentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($currentUser)
            ->event('Visit - CILs')
            ->log($currentUser->name . ' visited CILs ');
    }

    public function buildCILsQuery(): Builder
    {
        if (auth()->user()->type == 'admin') {
            return Lead::query()->where(['is_delay' => 0, 'type' => 'cil']);
        }
        
        return $this->buildNonAdminCILsQuery();
    }

    private function buildNonAdminCILsQuery(): Builder
    {
        $users = User::get();
        $currentUser = auth()->user();
        $userChildren = $this->getUserChildrenArray($currentUser, $users);
        $leadBulk = $this->buildLeadBulk($currentUser, $userChildren);
        
        return Lead::query()
            ->where('is_delay', 0)
            ->whereIn('id', $leadBulk);
    }

    private function getUserChildrenArray($currentUser, $users): array
    {
        $userChildren = [];
        foreach ($users as $user) {
            $parents = $user->getParents()->pluck('id')->toArray();
            if (in_array($currentUser->id, $parents)) {
                $userChildren[] = [
                    'id' => $user->id,
                    'name' => $user->name,
                ];
            }
        }
        return $userChildren;
    }

    private function buildLeadBulk($currentUser, $userChildren): array
    {
        $leadBulk = [];
        
        // Add current user's leads
        if (count($currentUser->leads) != 0) {
            foreach ($currentUser->leads as $lead) {
                $leadBulk[] = $lead->id;
            }
        }
        
        // Add children's leads
        foreach ($userChildren as $child) {
            $sales = User::find($child['id']);
            if (count($sales->leads) != 0) {
                foreach ($sales->leads as $lead) {
                    $leadBulk[] = $lead->id;
                }
            }
        }
        
        return array_unique($leadBulk);
    }

    public function applyCILsFilters(Builder $leads, Request $request): void
    {
        $filters = [
            'user_id' => 'user_id',
            'creator_id' => 'created_by',
            'status_id' => 'status_id',
            'contact_tool' => 'contact_tool',
            'campaign_id' => 'campaign_id',
            'reply_option' => 'reply_option',
            'purpose' => 'purpose',
            'segment' => 'segment',
            'tag' => 'tag',
        ];

        foreach ($filters as $requestKey => $columnName) {
            if ($request->$requestKey) {
                $leads->whereIn($columnName, $request->$requestKey);
            }
        }

        // Time filters
        if ($request->time1 && $request->time2) {
            $leads->whereTime('created_at', '>=', $request->time1)
                ->whereTime('created_at', '<=', $request->time2);
        }

        if ($request->date1 && $request->date2) {
            $leads->whereDate('created_at', '>=', $request->date1)
                ->whereDate('created_at', '<=', $request->date2);
        }

        if ($request->start_date && $request->end_date) {
            $leads->whereDate('confirmation_date', '>=', $request->start_date)
                ->whereDate('confirmation_date', '<=', $request->end_date);
        }
    }

    public function getCILsFilterData(): array
    {
        return [
            'countries' => Country::get(),
            'statuses' => Status::get(),
            'users' => User::all(),
            'creators' => User::all(),
            'campaigns' => Campaign::all(),
        ];
    }

    public function getUserChildrenForCILs()
    {
        $currentUser = auth()->user();
        $users = User::all();
        
        if ($currentUser->type != 'admin') {
            return $this->getUserChildrenArray($currentUser, $users);
        }
        
        return $users;
    }

    public function calculateMaxPhoneNumbers($leads): int
    {
        $leadNumbers = 0;
        foreach ($leads as $lead) {
            $maxCount = count($lead->phone_numbers);
            if ($maxCount > $leadNumbers) {
                $leadNumbers = $maxCount;
            }
        }
        return $leadNumbers;
    }

    public function logLeadsIndexActivity(): void
    {
        $model = new Lead();
        $currentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($currentUser)
            ->event('Visit - Leads Index')
            ->log($currentUser->name . ' viewed leads index ');
    }
} 