<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Laratrust\Traits\LaratrustUserTrait;
use Illuminate\Database\Eloquent\SoftDeletes;

class User extends Authenticatable
{
    use LaratrustUserTrait;
    use HasApiTokens, HasFactory, Notifiable;
    use SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $guarded = [];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

    public function created_leads()
    {
        return $this->hasMany(Lead::class, 'id', 'created_by');
    }

    public function assigned_leads()
    {
        return $this->hasMany(Lead::class, 'id', 'user_id');
    }

    public function tickets()
    {
        return $this->belongsToMany(Ticket::class, 'ticket_user');
    }

    public function user_tickets()
    {
        return $this->hasMany(Ticket::class, 'user_id');
    }

    public function projects()
    {
        return $this->belongsToMany(Project::class);
    }

    public function leads()
    {
        return $this->hasMany(Lead::class);
    }

    public function Ticket_creator()
    {
        return $this->hasMany(Ticket::class, 'created_by');
    }

    public function ticket_actions()
    {
        return $this->hasMany(Action::class, 'user_id');
    }

    public function phases()
    {
        return $this->belongsToMany(Phase::class);
    }

    public function clients()
    {
        return $this->hasMany(Client::class);
    }

    public function logs()
    {
        return $this->hasMany(ActivityLog::class);
    }

    public function units()
    {
        return $this->hasMany(Unit::class);
    }

    public function actions()
    {
        return $this->hasMany(Action::class, 'user_id');
    }

    public function comments()
    {
        return $this->hasMany(Comment::class);
    }

    public function events()
    {
        return $this->hasManyThrough(
            Event::class,
            Communication::class,
            'user_id',
            'communication_id',
            'id',
            'id'
        );
    }

    public function alerts()
    {
        return $this->hasMany(Alert::class);
    }

    public function latestAlerts()
    {
        return $this->hasMany(Alert::class)->latest()->take(5)->orderBy('id', 'desc');
    }

    public function batches()
    {
        return $this->hasMany(Batch::class);
    }

    public function role()
    {
        return $this->belongsTo(Role::class, 'role_id');
    }

    public function phones()
    {
        return $this->hasMany(PhoneNumber::class);
    }

    public function parent()
    {
        return $this->hasOne(User::class, 'id', 'assigned_to');
    }

    public function childs()
    {
        return $this->hasMany(User::class, 'assigned_to');
    }

    public function streams()
    {
        return $this->hasMany(Stream::class);
    }

    public function inquiries()
    {
        return $this->hasMany(Inquiry::class);
    }

    public function getParents()
    {
        $parents = collect([]);

        $parent = $this->parent;

        while (!is_null($parent)) {
            $parents->push($parent);
            $parent = $parent->parent;
        }

        return $parents;
    }

    public function allChildren()
    {
        $children = $this->childs;

        foreach ($this->childs as $child) {
            $children = $children->merge($child->allChildren());
        }

        return $children;
    }

    public function permissions()
    {
        return $this->belongsToMany(Permission::class);
    }

    public function created_by()
    {
        return $this->hasOne(User::class, 'id', 'created_by');
    }


    public function brokers()
    {
        return $this->hasMany(Broker::class);
    }

    public function accessed_clients()
    {
        return $this->belongsToMany(Client::class, 'client_user');
    }
}
