<?php

namespace App\Http\Controllers;

use App\Models\File;

use App\Models\Role;
use App\Models\User;
use App\Models\Phase;
use App\Models\Gallery;
// use App\Http\Requests\ClientRequest;
use App\Models\Project;
use App\Models\Facility;
use App\Models\PaymentPlan;
use App\Models\PaymentType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Requests\ProjectRequest;
use App\Http\Resources\ProjectResource;
use Illuminate\Support\Facades\Storage;


class ProjectController extends Controller
{
    // public function __construct()
    // {
    //     $this->middleware('permission:projects-create',['only' => ['create', 'store']]);
    //     $this->middleware('permission:projects-read',['only' => ['show']]);
    //     $this->middleware('permission:projects-update',['only' => ['edit', 'update']]);
    //     $this->middleware('permission:projects-delete',['only' => ['destroy']]);
    // }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if (auth()->user()->type == 'admin') {
            $allProjects = Project::get();
        } else {
            $allProjects = auth()->user()->projects->where('is_active', 1);
        }
        $user = auth()->user();

        activity()
            ->event('visit')
            ->log($user->name . ' accessed projects index');

        $projects = Project::query();

        if ($request->project_id) {
            $projects = Project::whereIn('id', $request->project_id);
        }

        if ($request->time1 && $request->time2) {
            $projects->whereTime('created_at', '>=', $request->time1)
                ->whereTime('created_at', '<=', $request->time2);
        }

        if ($request->date1 && $request->date2) {
            $projects->whereDate('created_at', '>=', $request->date1)
                ->whereDate('created_at', '<=', $request->date2);
        }

        $projects = $projects->get();


        return view('projects.index', compact('projects', 'allProjects'));
    }

    public function getProjects()
    {
        //
        try{
            $projects = Project::get();
            $data = ProjectResource::collection($projects);

            return response()->json([
                'statusCode' => 200,
                'data' => $data
            ]);
        }catch(Exception $e){
            return response()->json([
                'statusCode' => 500,
                'message' => $e->getMessage(),
            ]);

        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $users = User::all();
        $roles = Role::all();
        $facilities = Facility::all();
        $paymentTypes = PaymentType::all();
        $user = auth()->user();
        $maintenance = PaymentPlan::where('type', 'maintenance')->get();
        $total_unit = PaymentPlan::where('type', 'total_unit')->get();
        activity()
            ->event('visit')
            ->log($user->name . ' projects creation page ');
        return view('projects.create', compact('facilities', 'users', 'roles', 'paymentTypes', 'maintenance', 'total_unit'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(ProjectRequest $request)
    {

        $project = new Project;
        $project->name = $request->name;
        $project->description = $request->description;
        $project->is_active = $request->is_active;
        $project->is_residential = $request->is_residential ?? 0;
        $project->mentainance = $request->mentainance;

        if ($request->hasFile('logo')) {
            $imageName = time() . '.' . $request->logo->extension();
            $request->logo->move(public_path('images'), $imageName);
            $project->logo = $imageName;
        }
        $project->save();

        if ($request->user_id) {
            $project->users()->attach($request->user_id);
        }
        // dd($project);
        if ($request->facility_id) {
            foreach ($request->facility_id as $id) {
                $project->facilities()->attach($id);
            }
        }

        if ($request->tu_payment_plan_id) {
            $project->PaymentPlans()->attach($request->tu_payment_plan_id, ['is_active' => 1]);
        }

        if ($request->m_payment_plan_id) {
            $project->PaymentPlans()->attach($request->m_payment_plan_id,  ['is_active' => 1]);
        }

        if ($request->role_id) {
            $project->roles()->attach($request->role_id);

            $roles = Role::whereIn('id', $request->role_id)->get();
            foreach ($roles as $role) {
                $project->users()->attach($role->users);
            }
        }
        $user = auth()->user();
        activity()
            ->event('create')
            ->log($user->name . ' created project: ' . $project->name);
        return redirect()->route('projects.index')->with('success', 'Project Created');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Project $project)
    {
        $user = auth()->user();

        // Phases
        if($user->type == 'admin'){
            $phases = $project->phases;
        }else{
            $phases = $user->phases()->where(['project_id'=> $project->id, 'is_active'=> 1])->get();
        }

        activity()
            ->event('visit')
            ->log($user->name . ' accsessed ' . $project->name . 'page');
        return view('projects.show', compact('project', 'phases'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Project $project)
    {
        $users = User::all();
        $roles = Role::all();
        $facilities = Facility::get();
        $projectFacilities = $project->facilities->pluck('id')->toArray();

        $projectUsers = $project->users->pluck('id')->toArray();
        $projectRoles = $project->roles->pluck('id')->toArray();

        $maintenance = PaymentPlan::where('type', 'maintenance')->get();
        $total_unit = PaymentPlan::where('type', 'total_unit')->get();

        $user = auth()->user();
        activity()
            ->event('visit')
            ->log($user->name . ' accessed project: ' . $project->name . 'Edit page');
        return view('projects.edit', compact('project', 'users', 'roles', 'projectUsers', 'projectRoles', 'facilities', 'projectFacilities', 'maintenance', 'total_unit'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Project $project)
    {
        $project->name = $request->name;
        $project->description = $request->description;
        $project->is_residential = $request->is_residential == '' ? 0 : 1;
        $project->mentainance = $request->mentainance;

        if ($request->hasFile('logo')) {
            // dd($request->logo);
            $imageName = time() . '.' . $request->logo->extension();
            $request->logo->move(public_path('images'), $imageName);
            $project->logo = $imageName;
        }
        $project->save();


        if ($request->user_id) {
            $project->users()->sync($request->user_id);
        }

        if ($request->facility_id) {
            $project->facilities()->sync($request->facility_id);
        } else {
            $project->facilities()->detach();
        }

        // if($request->tu_payment_plan_id)
        // { 
        //     $project->PaymentPlans()->attach($request->tu_payment_plan_id, ['is_active'=>1]);
        // }

        // if($request->m_payment_plan_id)
        // {
        //     $project->PaymentPlans()->attach($request->m_payment_plan_id,  ['is_active'=>1]);
        // }

        if (isset($request->role_id)) {
            $project->roles()->sync($request->role_id);

            $roles = Role::whereIn('id', $request->role_id)->get();
            foreach ($roles as $role) {
                $project->users()->sync($role->users);
            }
        }

        $user = auth()->user();
        activity()
            ->event('update')
            ->log($user->name . ' updated project: ' . $project->name);
        return redirect()->route('projects.show', $project->id)->with('success', 'Project Updated');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Project $project)
    {
        // dd($project);
        if (count($project->phases) == 0) {
            $user = auth()->user();
            activity()
                ->event('delete')
                ->log($user->name . ' deleted Phase: ' . $project->name);

            $project->delete();

            return redirect()->route('projects.index')
                ->with('success', 'Project deleted successfully');
        } else {
            return redirect()->route('projects.index')
                ->with('error', "This Projesct has Phases, You can't delete this project");
        }
    }

    public function deletePermanently($id)
    {
        $project = Project::withTrashed()->find($id);
        $project->forceDelete();
        return redirect()->route('projects.index')->with('success', 'Project Trashed successfully');
    }

    public function hideProject($id)
    {
        $project = Project::find($id);
        $user = auth()->user();
        // dd($project);
        if ($project->is_active == 0) {
            $project->is_active = 1;
            $project->save();
            activity()
                ->event('hide')
                ->log($user->name . ' Activated : ' . $project->name);
        } else {
            $project->is_active = 0;
            $project->save();
            activity()
                ->event('hide')
                ->log($user->name . 'de-Activated : ' . $project->name);
        }
        foreach ($project->phases()->get() as $phase) {
            foreach ($phase->buildings()->get() as $building) {
                foreach ($building->units()->get() as $unit) {
                    $unit->is_active = $project->is_active;
                    $unit->save();
                }
                $building->is_active = $project->is_active;
                $building->save();
            }
            $phase->is_active = $project->is_active;
            $phase->save();
        }
        return redirect()->route('projects.index')->with('success', $project->is_active == 0 ? 'Project has been hidden successfully' : 'Project Activated Successfully');
    }

    public function fileUpload(Request $request)
    {
        // dd(count($request['files']));
        $project = Project::find($request->project_id);
        if (isset($request['files'])) {

            for ($i = 0; $i < count($request['files']); $i++) {

                $fileName = time() . $request['files'][$i]->getClientOriginalName();
                $extension = $request['files'][$i]->extension();

                if ($extension == 'csv' || $extension == 'txt'  || $extension == 'xlsx'  || $extension == 'pdf' || $extension == 'docx') {
                    // dd($extension);
                    $request['files'][$i]->move(public_path('files'), $fileName);

                    /* Store $fileName name in DATABASE from HERE */
                    $file = new File();
                    $file->project_id = $request->project_id;
                    $file->file = $fileName;
                    $file->save();
                } else {
                    return redirect()->back()->with('error', 'it not allowed file');
                }
            }
            return redirect()->route('projects.show', $project->id)->with('success', 'Files uploaded successfully');
        }

        if (isset($request['images'])) {
            for ($i = 0; $i < count($request['images']); $i++) {

                $imageName = time() . $request['images'][$i]->getClientOriginalName();
                $extension = $request['images'][$i]->extension();
                // dd($extension);
                if ($extension == 'png' || $extension == 'jpg' || $extension == 'jpeg') {
                    $request['images'][$i]->move(public_path('images'), $imageName);

                    /* Store $imageName name in DATABASE from HERE */
                    $image = new Gallery();
                    $image->project_id = $request->project_id;
                    $image->name = $imageName;
                    $image->save();
                } else {
                    return redirect()->back()->with('error', 'it not allowed image');
                }
            }
            return redirect()->route('projects.show', $project->id)->with('success', 'Files uploaded successfully');
        }
    }

    public function archive()
    {
        // dd('projects');

        $projects = Project::onlyTrashed()->get();
        // Log activity
        $model = new Project();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Projects archive')
            ->log(auth()->user()->name . ' viewed projects archive');

        return view('projects.archive', compact('projects'));
    }

    public function restore($ProjectId)
    {
        Project::withTrashed()->find($ProjectId)->restore();
        $project = Project::find($ProjectId);

        $user = auth()->user();

        activity()
            ->performedOn($project)
            ->causedBy($user)
            ->event('Restore Project')
            ->log('Project (' . $project->name . ') Restored by ' . $user->name);


        return redirect()->back();
    }

    public function restoreAll()
    {
        Project::onlyTrashed()->restore();

        $project = new Project();
        $user = auth()->user();

        activity()
            ->performedOn($project)
            ->causedBy($user)
            ->event('Restore All projects')
            ->log('All project Restored by ' . $user->name);

        return redirect()->back();
    }

    public function deleteImage($id)
    {
        Gallery::find($id)->delete();
        return redirect()->back();
    }
}
