<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Phase;
use App\Models\Project;
use App\Models\Building;
use App\Exports\PhasesExport;
use Doctrine\Inflector\Rules\English\Rules;
// use App\Http\Requests\ClientRequest;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;


class PhaseController extends Controller
{
    // public function __construct()
    // {
    //     $this->middleware('permission:phases-create',['only' => ['create', 'store']]);
    //     $this->middleware('permission:phases-read',['only' => ['show']]);
    //     $this->middleware('permission:phases-update',['only' => ['edit', 'update']]);
    //     $this->middleware('permission:phases-delete',['only' => ['destroy']]);
    // }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $phases = Phase::query();
        $user = auth()->user();
        if($user->type == 'admin' || auth()->user()->role->name == 'Operation Specialist')
        {
            $phases = $phases;
        }
        else
        {
            $phases = $phases->whereHas('users',function($q) use($user){
                $q->where('users.id',$user->id);
            });
        }
        
        $user = auth()->user();
        activity()
        ->event('visit')
        ->log($user->name .' visited Phases index page ');


        if($request->phase_id){
            $phases = Phase::whereIn('id',$request->phase_id);
        }
        if($request->project_id){
            $phases->whereIn('project_id', $request->project_id);
        }

        if($request->time1 && $request->time2){
            $phases->whereTime('created_at', '>=',$request->time1)
            ->whereTime('created_at','<=',$request->time2);
        }
        
        if($request->date1 && $request->date2){
            $phases->whereDate('created_at', '>=',$request->date1)
            ->whereDate('created_at','<=',$request->date2);
        }

        $phases = $phases->paginate(100)->withQueryString();
        $projects = Project::get();
        return view('phases.index', compact('phases', 'projects'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $user = auth()->user();
        $project = Project::find($request->id);
        $users = $project->users;
        activity()
        ->event('visit')
        ->log($user->name .' visited Phase create page ');
        return view('phases.create', compact('project', 'users'));

    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $phase = new Phase();
        $phase->name = $request->name;
        $phase->project_id = $request->project_id;

        if($phase->project->is_residential == 0){
            $phase->type = $request->type;
        }
        $phase->save();
 
        if($request->user_id){
            $phase->users()->attach($request->user_id);
        }
        $user = auth()->user();
     
        activity()
        ->event('create')
        ->log($user->name .' created Phase: ' . $request->name);
        return redirect()->route('projects.show', $request->project_id)
                        ->with('success','Phase created successfully.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Phase $phase)
    {
        $user = auth()->user();
        activity()
        ->event('visit')
        ->log($user->name .' visited Phase: ' . $phase->name);

        return view('phases.show' , compact('phase'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Phase $phase)
    {
        //
        $projects = Project::get();
        $users = $phase->project->users;
        $phaseUsers = $phase->users->pluck('id')->toArray(); 
        
        $user = auth()->user();
        activity()
        ->event('visit')
        ->log($user->name .' visited Phase: ' . $phase->name . 'edit page');
        
        
        return view('phases.edit' , compact('phase', 'projects','phaseUsers', 'users'));
    }
    
    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Phase $phase)
    {
        //
        $phase->name = $request->name;
        $phase->project_id = $request->project_id;
        
        if($phase->project->is_residential == 0){
            $phase->type = $request->type;
        }
        
        $phase->save();
        if($request->user_id){
            $phase->users()->sync($request->user_id);
        }
        
        $user = auth()->user();
        activity()
        ->event('update')
        ->log($user->name .' updated Phase: ' . $phase->name);
        
        return redirect()->route('phases.show', $phase->id)
                        ->with('success','Phase updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Phase $phase)
    {
        if(count($phase->buildings) == 0){

            $user = auth()->user();
            activity()
            ->event('delete')
            ->log($user->name .' deleted Phase: ' . $phase->name);
    
            $phase->delete();
    
            return redirect()->route('projects.show', $phase->project->id)
                        ->with('success','Phase deleted successfully');
        }else{
            return redirect()->route('projects.show', $phase->project->id)
                        ->with('error',"You Can't Delete This Phase, This Phase has Buildings");
        }
        
    }

    public function hidePhase($id)
    {
        //
        $phase = Phase::find($id);
        $user = auth()->user();

        if($phase->is_active == 0){
            $phase->is_active = 1;
            activity()
                ->event('hide')
                ->log($user->name .' Activated : ' . $phase->name);

        }else{
            $phase->is_active = 0;
            activity()
                ->event('hide')
                ->log($user->name .' de-Activated : ' . $phase->name);
        }
        $phase->save();

        foreach($phase->buildings()->get() as $building){
            foreach($building->units()->get() as $unit){
                $unit->is_active = $phase->is_active;
                $unit->save();
            }
            $building->is_active = $phase->is_active;
            $building->save();
        }

        return redirect()->back()->with('success', $phase->is_active == 0 ? 'phase has been hidden successfully' : 'phase Activated Successfully');
    }

    public function archive()
    {
        // dd('projects');
        
        $phases = Phase::onlyTrashed()->get();
        // Log activity
        $model = new Phase();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Phases archive')
            ->log(auth()->user()->name .' viewed Phases archive');

        return view('phases.archive',compact('phases'));
    }

    public function deletePermanently($id)
    {
        $phase = Phase::withTrashed()->find($id);
        $phase->forceDelete();

        return redirect()->route('phases.index')->with('success', 'Phase Trashed successfully');
    }

    public function restore($phasetId)
    {
        Phase::withTrashed()->find($phasetId)->restore();
        $phase = Phase::find($phasetId);

        $user = auth()->user();
            
        activity()
        ->performedOn($phase)
        ->causedBy($user)
        ->event('Restore phase')
        ->log('Phase (' . $phase->name . ') Restored by '. $user->name);


        return redirect()->back();
    }

    public function restoreAll()
    {
        Phase::onlyTrashed()->restore();

        $phase = new Phase();
        $user = auth()->user();
            
        activity()
        ->performedOn($phase)
        ->causedBy($user)
        ->event('Restore All phases')
        ->log('All phase Restored by '. $user->name);

        return redirect()->back();
    }

    /**
     * Export phases to Excel
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function export(Request $request)
    {
        $user = auth()->user();
        activity()
            ->event('export')
            ->log($user->name . ' exported phases data');

        return Excel::download(new PhasesExport($request), 'phases_' . date('Y-m-d_H-i-s') . '.xlsx');
    }
}
