<?php

namespace App\Http\Controllers;

use PDF;
use Auth;
use Exception;
use App\Models\User;
use App\Models\Agent;
use App\Models\File; 
use App\Models\Broker;
use App\Models\Country;
use App\Models\Gallery;
use App\Models\ActivityLog;
use App\Models\AgentBroker;
use App\Models\PhoneNumber;
use Illuminate\Http\Request;
use App\Imports\BrokerImport;
use App\Models\Communication;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Storage;
use App\Http\Requests\UploadBrokerRequest;



class BrokerController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        // Edit Activity Log
        $model = new Broker();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Broker Index')
            ->log(auth()->user()->name . ' viewed broker index ');
        // end Activity Log

        $brokers = Broker::query();
        $agents = Agent::get();
        $users = User::get();

        $sales_man = $request->salesMan_id;
        $created_at = $request->Created_at;
        $Created_to = $request->Created_to;


        if ($request->user_id) {
            $brokers->whereIn('user_id', $request->user_id);
        }

        // dd($request);
        if ($request->created_at_from && $request->created_at_to) {
            $brokers->whereBetween('created_at', [$request->created_at_from, $request->created_at_to]);
        }
        if ($request->updated_at_from && $request->updated_at_to) {
            $brokers->whereBetween('updated_at', [$request->updated_at_from, $request->updated_at_to]);
        }
        if ($request->broker_id) {
            $brokers->whereIn('id', $request->broker_id);
        }
        if ($request->status) {
            $brokers->whereIn('status', $request->status);
        }

        
        $totalBrokers = $brokers->get();
        $brokers = $brokers->paginate(50)->withQueryString(); 
        $brokerNumbers = 0;
        foreach ($brokers as $broker) {
            $maxCount = count($broker->phone_numbers);
            // dd($maxCount);
            if ($maxCount > $brokerNumbers) {
                $brokerNumbers = $maxCount;
            }
        }
        $model = new Broker();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Brokers')
            ->log(auth()->user()->name . ' visited brokers');
        
        return view('brokers.index', compact('brokers', 'brokerNumbers', 'totalBrokers', 'agents', 'users'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        // Edit Activity Log
        $model = new Broker();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Broker Create')
            ->log(auth()->user()->name . ' visited broker create');
        // end Activity Log

        $users = User::get();
        $countries = Country::get();
        return view('brokers.create', compact('users', 'countries'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // Edit Activity Log
        $model = new Broker();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Store - Broker')
            ->log(auth()->user()->name . ' store broker' . '(' . $request->name . ')');
        // end Activity Log

        $broker = new Broker();
        $broker->name = $request->name;
        $broker->description = $request->description;
        $broker->email = $request->email;
        $broker->type = $request->type;
        $broker->address = $request->address;
        $broker->owner_name  = $request->owner_name;
        $broker->commercial_record = $request->commercial_record;
        $broker->authorized_name = $request->authorized_name;
        $broker->sales_admin = $request->sales_admin;
        $broker->user_id = $request->user_id;
        if ($request->has_contract) {
            $broker->has_contract = 1;
            $broker->contract_startDate = $request->contract_startDate;
            $broker->contract_endDate = $request->contract_endDate;
            $broker->commission = $request->commission;
        }
        if($request->status == 'pending'){
            $broker->status = 'pending';
        }else{
            $broker->status = 'accepted';
        }
        $broker->save();
        // Store Phone Numbers of client
        try {
            if ($request->mobile_number) {
                for ($i = 0; $i < count($request->mobile_number); $i++) {
                    $phone = new PhoneNumber();
                    $phone->broker_id = $broker->id;
                    $phone->country_id = $request->country_id[$i];
                    $phone->phone = $request->mobile_number[$i];
                    $phone->save();
                }
            }
        } catch (Exception $e) {
            $broker->forceDelete();
            return redirect()->back()->withErrors(['msg' => 'This Phone Number is Duplicated']);
        }

        // upload files
        if (isset($request['files'])) {

            for ($i = 0; $i < count($request['files']); $i++) {

                $fileName = time() . $request['files'][$i]->getClientOriginalName();
                $extension = $request['files'][$i]->extension();

                if ($extension == 'csv' || $extension == 'txt'  || $extension == 'xlsx'  || $extension == 'pdf' || $extension == 'docx') {
                    // dd($extension);
                    $request['files'][$i]->move(public_path('files'), $fileName);

                    /* Store $fileName name in DATABASE from HERE */
                    $file = new File();
                    $file->broker_id = $broker->id;
                    $file->file = $fileName;
                    $file->save();
                } else {
                    return redirect()->back()->with('error', 'it not allowed file');
                }
            }
        }

        // upload image
        if (isset($request['images'])) {
            for ($i = 0; $i < count($request['images']); $i++) {

                $imageName = time() . $request['images'][$i]->getClientOriginalName();
                $extension = $request['images'][$i]->extension();
                // dd($extension);
                if ($extension == 'png' || $extension == 'jpg' || $extension == 'jpeg') {
                    $request['images'][$i]->move(public_path('images'), $imageName);

                    /* Store $imageName name in DATABASE from HERE */
                    $image = new Gallery();
                    $image->broker_id = $broker->id;
                    $image->name = $imageName;
                    $image->save();
                } else {
                    return redirect()->back()->with('error', 'it not allowed image');
                }
            }
        }
        $logText = auth()->user()->name .' (created broker) '. $broker->name; 
        $model = new Broker();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('create')
            ->log($logText);

        if($request->is_broker){
            return view('brokers.thanks')->with('success', 'Thanks');
        }else{
            return redirect()->route('brokers.show', $broker->id)->with('success', 'broker created successfully');
        }

    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Broker  $broker
     * @return \Illuminate\Http\Response
     */
    public function show(Broker $broker)
    {
        // Edit Activity Log
        $model = new Broker();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Show - Broker')
            ->log(auth()->user()->name . ' show broker' . '(' . $broker->name . ')');
        // end Activity Log
        $activities = ActivityLog::where(['subject_id' => $broker->id, 'subject_type' => 'App\Models\Broker'])->orderBy('created_at', 'asc')->get();
// dd($activities);
        $codes = Country::get();
        $users = User::get();
        return view('brokers.show', compact('broker', 'codes', 'activities', 'users'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Broker  $broker
     * @return \Illuminate\Http\Response
     */
    public function edit(Broker $broker)
    {
        // Edit Activity Log
        $model = new Broker();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Broker Edit')
            ->log(auth()->user()->name . 'visited broker edit');
        // end Activity Log

        $users = User::get();
        return view('brokers.edit', compact('broker', 'users'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Broker  $broker
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Broker $broker)
    {
        // Edit Activity Log
        $model = new Broker();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Update - Broker')
            ->log(auth()->user()->name . ' updated broker' . '(' . $request->name . ')');
        // end Activity Log

        $broker->name = $request->name;
        $broker->description = $request->description;
        $broker->email = $request->email;
        $broker->type = $request->type;
        $broker->address = $request->address;
        $broker->owner_name     = $request->owner_name;
        $broker->commercial_record = $request->commercial_record;
        $broker->authorized_name = $request->authorized_name;
        $broker->sales_admin = $request->sales_admin;
        $broker->user_id = $request->user_id;
        if ($request->has_contract) {
            $broker->has_contract = 1;
            $broker->contract_startDate = $request->contract_startDate;
            $broker->contract_endDate = $request->contract_endDate;
            $broker->commission = $request->commission;
        } else {
            $broker->has_contract = 0;
            $broker->contract_startDate = null;
            $broker->contract_endDate = null;
            $broker->commission = null;
        }

        $broker->save();
        return redirect()->route('brokers.show', $broker->id)->with('success', 'broker updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Broker  $broker
     * @return \Illuminate\Http\Response
     */
    public function destroy(Broker $broker)
    {
        // Edit Activity Log
        $model = new Broker();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Delete - Broker')
            ->log(auth()->user()->name . ' deleted broker' . '(' . $broker->name . ')');
        // end Activity Log

        $broker->delete();
        return redirect()->back()->with('success', 'broker Archived successfully');
    }

    public function fileUpload(Request $request)
    {
        $broker = Broker::find($request->broker_id);

        // Edit Activity Log
        $model = new Broker();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Upload - Broker File')
            ->log(auth()->user()->name . ' uploaded broker file on'  . '(' . $broker->name . ')');
        // end Activity Log


        if (isset($request['files'])) {
            for ($i = 0; $i < count($request['files']); $i++) {
                $fileName = time() . $request['files'][$i]->getClientOriginalName();
                $extension = $request['files'][$i]->extension();

                if ($extension == 'csv' || $extension == 'txt'  || $extension == 'xlsx'  || $extension == 'pdf' || $extension == 'docx') {
                    // dd($extension);
                    $request['files'][$i]->move(public_path('files'), $fileName);

                    /* Store $fileName name in DATABASE from HERE */
                    $file = new File();
                    $file->broker_id = $request->broker_id;
                    $file->file = $fileName;
                    $file->save();
                } else {
                    return redirect()->back()->with('error', 'it not allowed file');
                }
            }
            return redirect()->route('brokers.show', $broker->id)->with('success', 'Files uploaded successfully');
        }
        if (isset($request['images'])) {
            for ($i = 0; $i < count($request['images']); $i++) {

                $imageName = time() . $request['images'][$i]->getClientOriginalName();
                $extension = $request['images'][$i]->extension();
                // dd($extension);
                if ($extension == 'png' || $extension == 'jpg' || $extension == 'jpeg') {
                    $request['images'][$i]->move(public_path('images'), $imageName);

                    /* Store $imageName name in DATABASE from HERE */
                    $image = new Gallery();
                    $image->broker_id = $request->broker_id;
                    $image->name = $imageName;
                    $image->save();
                } else {
                    return redirect()->back()->with('error', 'it not allowed image');
                }
            }
            return redirect()->route('brokers.show', $broker->id)->with('success', 'Images uploaded successfully');
        }
    }

    public function brokersArchive()
    {
        $brokers = Broker::onlyTrashed()->get();

        // Activity Log
        $role = new Broker();
        $user = auth()->user();

        activity()
            ->performedOn($role)
            ->causedBy($user)
            ->event('Visit Broker Archive')
            ->log('User (' . $user->name . ') Visited Broker Archive Page ');


        return view('brokers.archive', compact('brokers'));
    }

    public function restoreAll()
    {
        Broker::onlyTrashed()->restore();

        $brokers = new Broker();
        $user = auth()->user();

        activity()
            ->performedOn($brokers)
            ->causedBy($user)
            ->event('Restore All Brokers')
            ->log('All brokers Restored by ' . $user->name);

        return redirect()->back();
    }

    public function restore($brokerId)
    {
        Broker::withTrashed()->find($brokerId)->restore();
        $broker = Broker::find($brokerId);

        $user = auth()->user();
        activity()
            ->performedOn($broker)
            ->causedBy($user)
            ->event('Restore broker')
            ->log('broker (' . $broker->name . ') Restored by ' . $user->name);

        return redirect()->route('brokers.index');
    }

    public function deletePermanently($id)
    {
        // Edit Activity Log
        $model = new Broker();
        $CurrentUser = auth()->user();
        $broker = Broker::withTrashed()->find($id);
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Delete - Archived Broker')
            ->log(auth()->user()->name . ' deleted archived broker' . '(' . $broker->name . ')');
        // end Activity Log

        foreach($broker->files as $file){
            $file->delete();
        }
        foreach($broker->images as $image){
            $image->delete();
        }

        foreach($broker->phone_numbers as $phone){
            $phone->delete();
        }
        $broker->forceDelete();

        return redirect()->route('brokers.index')->with('success', 'Broker Trashed successfully');
    }

    public function deleteImage($id)
    {
        Gallery::find($id)->delete();
        return redirect()->back();
    }


    public function importBrokers(Request $request)
    {
        try{
        // dd('some');
            if ($request->file) {   
                // dd($request->file);
                Excel::import(new BrokerImport, $request->file);

                $model = new Broker();
                $CurrentUser = auth()->user();
                
                activity()
                ->performedOn($model)
                ->causedBy($CurrentUser)
                ->event('Broker - Import')
                ->log(auth()->user()->name . ' Imported Brokers Sheet');

                return redirect()->route('brokers.index')->with('success', 'Broker Uploaded Successfully');
            } else {
                return redirect()->back()->with('error', 'Choose File Please');
            }
        }catch(Exception $e){
            Log::info($e->getMessage());
        }
        
    }

    public function brokerBulkAction(Request $request, $brokers)
    {
        
        try{
            $brokerIds = explode(",",$brokers);
            $agent = agent::find($request->agent_id);
            $IDs = array();
            foreach($brokerIds as $id){
                $agentBroker = AgentBroker::where(['broker_id'=> $id , 'agent_id' => $agent->id])->get();
                if(count($agentBroker) == 0){
                    array_push($IDs, $id);
                }
            }
            $agent->brokers()->attach($IDs);
            
            return redirect()->route('brokers.index')->with('success', 'Agent Assigned to Broker Successfully');
        }catch(Exception $e){
            return redirect()->back()->withErrors(['msg' => $e->getMessage()]);
        }
    }
    

    public function brokerRegistration()
    {
        $countries = Country::get();
        return view('brokers.registration', compact('countries'));
    }

    public function activityManagment(Request $request)
    {
        // activity()
        // ->event('visit')
        // ->log(auth()->user()->name . 'Events Index');

        $communications = Communication::query();
        $communications->whereNotNull('broker_id');

        $creators = User::get();
        $brokers = Broker::get();
        $agents = Agent::get();

        // dd($request->creators_id);
        if ($request->creators_id) {
            $communications->whereIn('user_id', $request->creators_id);
        }

        if ($request->brokers_id) {
            $communications->whereIn('broker_id', $request->brokers_id);
        }

        if ($request->type) {
            $communications->whereIn('type', $request->type);
        }
        if ($request->state) {
            $communications->whereIn('state', $request->state);
        }
        if ($request->agent_id) {
            $communications->whereIn('agent_id', $request->agent_id);
        }


        if ($request->time1 && $request->time2) {
            $communications->whereTime('created_at', '>=', $request->time1)
                ->whereTime('created_at', '<=', $request->time2);
        }

        if ($request->date1 && $request->date2) {
            $communications->whereDate('created_at', '>=', $request->date1)
                ->whereDate('created_at', '<=', $request->date2);
        }

        $CurrentUser = auth()->user();

        if ($CurrentUser->type != 'admin') {
            $UserChildren = array($CurrentUser->id);
            foreach ($creators as $user) {
                $parents = $user->getParents()->pluck('id')->toArray();
                if (in_array($CurrentUser->id, $parents)) {
                    array_push($UserChildren, $user->id);
                }
            }

            $communications->whereIn('user_id', $UserChildren);
        } else {
            $UserChildren = $creators;
        }
        
        $totalActions = $communications->get();
        $communications = $communications->paginate(50)->withQueryString();

        return view('brokers.activities', compact('communications', 'creators', 'brokers','totalActions', 'agents'));
    }

    public function brokerAgents(Request $request)
    {

        $broker = Broker::find($request->broker_id);
        $agents = $broker->agents;
        return json_encode(['agents' => $agents]);
    }

    public function brokerStatus(Request $request)
    {
        $broker = Broker::find($request->broker_id);
        $broker->status = $request->status;
        $broker->save();

        $logText = auth()->user()->name .' '. $broker->status . ' ' . $broker->name;
        $model = new Broker();
                $CurrentUser = auth()->user();
                
                activity()
                ->performedOn($model)
                ->causedBy($CurrentUser)
                ->event('Broker - Import')
                ->log($logText);

        return redirect()->back()->with('success', 'Broker ' . $broker->status . ' successfully');

    }

    public function brokerReAssign(Request $request)
    {
        $broker = Broker::find($request->broker_id);
        $user = User::find($request->user_id);
        $broker->user_id = $request->user_id;
        $broker->save();

        $logText = auth()->user()->name .' Re-assigned '. $broker->name . ' To ' . $user->name;
        $model = new Broker();
        $CurrentUser = auth()->user();
        
        activity()
        ->performedOn($model)
        ->causedBy($CurrentUser)
        ->event('Broker - reAssign sales')
        ->log($logText);

        return redirect()->back()->with('success', 'Broker Re-Assigned to '. $user->name . ' successfully');

    }

    public function export(Request $request)
    {
        $user = auth()->user();
        activity()
            ->event('export')
            ->log($user->name . ' exported brokers data');

        return Excel::download(new \App\Exports\BrokersExport($request), 'brokers_' . date('Y-m-d_H-i-s') . '.xlsx');
    }
}