<?php

namespace App\Http\Controllers\Administration;

use Illuminate\Http\Request;
use App\Http\Requests\RoleRequest;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use App\Models\Role;
use App\Models\Permission;
use Exception;

class RoleController extends Controller
{
    // public function __construct()
    // {
    //     $this->middleware('permission:roles-create');
    //     $this->middleware('permission:roles-read'); 
    //     $this->middleware('permission:roles-update');
    //     $this->middleware('permission:roles-delete');
    // }

    public function index()
    {
        $roles = Role::get();
        $role = new Role();
        $user = auth()->user();

        activity()
            ->performedOn($role)
            ->causedBy($user)
            ->event('Visit Roles index')
            ->log('User (' . $user->name . ') Visited Roles Index Page ');


        return view('roles.index', compact('roles'));
    }

    public function rolesTree()
    {
        $allRoles = Role::get();
        $roles = Role::where('role_id', '=', null)->get();
        $parentCodes = Role::where('role_id', '!=', null)->get();

        $arrayOfCodes = [];
        foreach ($parentCodes as $code) {
            array_push($arrayOfCodes, $code->role_id);
        }
        // dd($arrayOfCodes);

        $role = new Role();
        $user = auth()->user();

        activity()
            ->performedOn($role)
            ->causedBy($user)
            ->event('Visit Roles index')
            ->log('User (' . $user->name . ') Visited Roles Tree Page ');

        return view('roles.tree', compact('roles', 'arrayOfCodes', 'allRoles'));
    }

    public function create()
    {
        $permissions = Permission::get();
        $roles = Role::get();

        $role = new Role();
        $user = auth()->user();

        activity()
            ->performedOn($role)
            ->causedBy($user)
            ->event('Visit Roles Create')
            ->log('User (' . $user->name . ') Visited Create Roles Page ');


        return view('roles.create', compact('permissions', 'roles'));
    }

    public function store(RoleRequest $request)
    {
        try {
            DB::beginTransaction();

            $role = new Role();
            $role->name = $request->name;
            $role->display_name = ucwords($request->name);
            $role->description = $request->description;
            $role->role_id = $request->role_id;
            $role->save();
            DB::commit();

            $user = auth()->user();

            activity()
                ->performedOn($role)
                ->causedBy($user)
                ->event('Create Role')
                ->withProperties([
                    'role_creator' => $user->name,
                    'created_role' => $request->name,
                    'creation_date' => $role->created_at,
                ])
                ->log('Role (' . $role->name . ') created by ' . $user->name);

            return redirect()->route('index.roles')->with('success', 'Role Created Successfully');
        } catch (Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function edit($roleId)
    {
        $role = Role::findOrFail($roleId);
        $permissions = Permission::get();
        $roles = Role::get();
        $role_permissions = $role->permissions->pluck('id')->toArray();

        $user = auth()->user();

        activity()
            ->performedOn($role)
            ->causedBy($user)
            ->event('Visit Roles Edit')
            ->log('User (' . $user->name . ') Visited Edit Roles Page ');


        return view('roles.edit', compact('role', 'permissions', 'role_permissions', 'roles'));
    }

    public function update(RoleRequest $request, $roleId)
    {
        $role = Role::findOrFail($roleId);
        try {
            DB::beginTransaction();

            $role->name = $request->name;
            $role->display_name = ucwords($request->name);
            $role->description = $request->description;
            $role->role_id = $request->role_id;
            $role->save();

            DB::commit();

            $user = auth()->user();

            activity()
                ->performedOn($role)
                ->causedBy($user)
                ->event('Update Role')
                ->withProperties([
                    'role_creator' => $user->name,
                    'updated_role' => $role->name,
                    'update_date' => $role->updated_at,
                ])
                ->log('Role (' . $role->name . ') updated by ' . $user->name);


            return redirect()->route('index.roles')->with('success', 'Role Updated Successfully');
        } catch (Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function show($roleId)
    {
        $role = Role::withTrashed()->find($roleId);

        $user = auth()->user();

        activity()
            ->performedOn($role)
            ->causedBy($user)
            ->event('Visit Role')
            ->log('User (' . $user->name . ') Visited Roles (' . $role->name . ') Page ');


        return view('roles.show', compact('role'));
    }

    public function destroy($roleId)
    {
        try {
            $role = Role::find($roleId);
            $role_name = $role->name;


            // Activity Log
            $user = auth()->user();

            activity()
                ->performedOn($role)
                ->causedBy($user)
                ->event('Delete Role')
                ->log('Role (' . $role_name . ') Deleted by ' . $user->name);

            $role->delete();

            return redirect()->route('index.roles')->with('success', 'Role Deleted Successfully');
        } catch (Exception $ex) {
            return $ex->getMessage();
        }
    }

    public function deletePermanently($id)
    {
        $role = Role::withTrashed()->find($id);
        // dd($role);
        $role->forceDelete();

        return redirect()->route('index.roles')->with('success', 'Role Trashed successfully');
    }

    public function Roles_Archive()
    {
        $roles = Role::onlyTrashed()->get();

        // Activity Log
        $role = new Role();
        $user = auth()->user();

        activity()
            ->performedOn($role)
            ->causedBy($user)
            ->event('Visit Roles Archive')
            ->log('User (' . $user->name . ') Visited Roles Archive Page ');


        return view('roles.archive', compact('roles'));
    }

    public function restore($id)
    {
        Role::withTrashed()->find($id)->restore();
        $role = Role::find($id);

        $user = auth()->user();

        activity()
            ->performedOn($role)
            ->causedBy($user)
            ->event('Restore Role')
            ->log('Role (' . $role->name . ') Restored by ' . $user->name);


        return redirect()->back();
    }

    public function restoreAll()
    {
        Role::onlyTrashed()->restore();

        $role = new Role();
        $user = auth()->user();

        activity()
            ->performedOn($role)
            ->causedBy($user)
            ->event('Restore All Roles')
            ->log('All Roles Restored by ' . $user->name);

        return redirect()->back();
    }
}
