<?php

namespace App\Http\Controllers\Administration;

use App\Helpers\LeadHelper;
use Exception;
use Carbon\Carbon;
use App\Models\Lead;
use App\Models\User;
use App\Models\Role;
use App\Models\Event;
use App\Models\Broker;
use App\Models\Source;
use App\Models\Status;
use App\Models\Country;
use App\Models\Inquiry;
use App\Models\Project;
use App\Models\Setting;
use App\Models\Campaign;
use App\Models\LeadUser;
use App\Models\ActivityLog;
use App\Models\PhoneNumber;
use App\Imports\LeadsImport;
use Illuminate\Http\Request;
use App\Models\Communication;
use App\Http\Requests\LeadRequest;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;
use App\Http\Requests\UploadLeadRequest;
use App\Exports\AllLeadsExport;
use App\Services\LeadFilterService;
use App\Services\CilService;
use App\Services\LeadService;

class LeadsController extends Controller
{
    public function __construct()
    {
        // $this->middleware('permission:leads-create', ['only' => ['create', 'store']]);
        // $this->middleware('permission:leads-read', ['only' => ['show']]);
        // $this->middleware('permission:leads-update', ['only' => ['edit', 'update']]);
        // $this->middleware('permission:leads-delete', ['only' => ['destroy']]);
        // $this->middleware('permission:leads-export', ['only' => ['export']]);
        // $this->middleware('permission:leads-import', ['only' => ['import']]);
        // $this->middleware('permission:leads-ChangeStatus',['only' => ['ChangeStatus']]);
        // $this->middleware('permission:leads-reAssignUser',['only' => ['reAssignUser']]);
    }

    public function index(Request $request, LeadFilterService $filterService, LeadService $leadService, $status = null)
    {
        $leads = $leadService->buildLeadsQuery();

        // Status filtering
        $statusMapping = [
            'fresh' => 1,
            'interested' => 20,
            'fakeNumber' => 21,
            'meeting' => 27,
            'pending' => 28,
            'followUp' => 29,
            'lowBudget' => 30,
            'noRegisteration' => 31,
            'differentLocation' => 32,
            'closed' => 36,
            'callFailed' => 37,
            'noAnswer' => 38,
        ];

        if (isset($statusMapping[$status])) {
            $leads->where('status_id', $statusMapping[$status]);
        }

        if (auth()->user()->type == 'admin' || auth()->user()->role->name == 'Marketing Team Leader' || auth()->user()->role->name == 'Marketing Manager' || auth()->user()->role->name == 'Sales Admin' || auth()->user()->role->name == 'Operation Specialist') {
            $leads->where(['is_delay' => 0, 'type' => 'lead']);
        } elseif (auth()->user()->assigned_to == null) {
            // $childUsers = auth()->user()->childs()->pluck('id')->toArray();
            $childUsers = auth()->user()->allChildren()->pluck('id')->toArray();
            if (count($childUsers) > 0) {
                $leads = Lead::where(function ($query) {
                    $query->where('created_by', auth()->user()->id)
                        ->orWhereIn('user_id', auth()->user()->allChildren()->pluck('id')->toArray());
                });
            } else {
                $leads = Lead::where('created_by', auth()->user()->id);
            }
        } else {
            $users = User::all();
            $CurrentUser = auth()->user();
            $UserChildren = array();

            foreach ($users as $userr) {
                $parents = $userr->getParents()->pluck('id')->toArray();
                if (in_array($CurrentUser->id, $parents)) {
                    array_push($UserChildren, [
                        'id' => $userr->id,
                        'name' => $userr->name,
                    ]);
                }
            }

            $LeadBulk = array();
            if ($CurrentUser->leads()->count() != 0) {
                foreach ($CurrentUser->leads as $lead) {
                    array_push($LeadBulk, $lead->id);
                }
            }
            for ($i = 0; $i < count($UserChildren); $i++) {
                $sales = User::find($UserChildren[$i]['id']);
                if (count($sales->leads) == 0) {
                    continue;
                } else {
                    foreach ($sales->leads as $lead) {
                        array_push($LeadBulk, $lead->id);
                    }
                }
            }
            $LeadBulk = array_unique($LeadBulk);
            // $leads = Lead::query();
            $leads->where('is_delay', 0);
            $leads->whereIn('id', $LeadBulk);
        }

        // Apply filters
        $leads = $filterService->applyFilters($request, $leads);

        // Fetch relationships
        $leads->with(['communications', 'phone_numbers']);

        $events = Event::with('communications')->get();

        // AJAX request handling
        if ($request->ajax()) {
            $count = $leads->count();
            $leads = $leads->paginate(100);

            $leadNumbers = $leadService->calculateMaxPhoneNumbers($leads);

            return response()->json([
                'html' => view('leads.partials.leads_table_paginated', [
                    'leads' => $leads,
                    'leadNumbers' => $leadNumbers,
                    'events' => $events,
                ])->render(),
                'total' => $count,
            ]);
        }

        $leads = $leads->paginate(100);
        $leadNumbers = $leadService->calculateMaxPhoneNumbers($leads);

        // Get filter data
        $filterData = $leadService->getFilterData();
        $UserChildren = $leadService->getUserChildrenForLeads();

        // Extract variables from filterData
        $statuses = $filterData['statuses'];
        $statusTypes = $filterData['statusTypes'];
        $users = $filterData['users'];
        $creators = $filterData['creators'];
        $campaigns = $filterData['campaigns'];
        $sources = $filterData['sources'];
        $projects = $filterData['projects'];
        $countries = $filterData['countries'];
        $brokers = $filterData['brokers'];

        // Activity Log
        $leadService->logActivity('Visit - Leads Index', auth()->user()->name . ' viewed leads index ');

        return view('leads.index', compact('leads', 'users', 'countries', 'campaigns', 'UserChildren', 'projects', 'sources', 'statuses', 'creators', 'leadNumbers', 'statusTypes', 'events', 'brokers'));
    }



    public function create()
    {
        $countries = Country::all();
        $campaigns = Campaign::all();
        $brokers = Broker::get();
        $users = User::get();
        $CurrentUser = auth()->user();
        if ($CurrentUser->type != 'admin') {
            $UserChildren = array();
            foreach ($users as $user) {
                $parents = $user->getParents()->pluck('id')->toArray();
                if (in_array($CurrentUser->id, $parents)) {
                    array_push($UserChildren, [
                        'id' => $user->id,
                        'name' => $user->name,
                    ]);
                }
            }
        } else {
            $UserChildren = $users;
        }

        $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Leads Create')
            ->log(auth()->user()->name . ' viewed leads creation page ');

        return view('leads.create', compact('UserChildren', 'countries', 'campaigns', 'brokers', 'users'));
    }

    public function store(Request $request, LeadService $leadService)
    {
        try {


            $lead = $leadService->createLead($request->all());
            
            try {
                if ($request->mobile_number) {
                    $leadService->createPhoneNumbers($lead, $request->mobile_number, $request->country_id);
                }
            } catch (Exception $e) {
                $lead->forceDelete();
                return redirect()->back()->withErrors(['msg' => 'This Phone Number is Duplicated']);
            }

            $leadService->logActivity('Create Lead', 'User(' . auth()->user()->name . ') created lead (' . $lead->name . ')', $lead);
            
            if ($lead->type == 'lead') {
                return redirect()->route('leads.index')->with('success', 'Lead Created Successfully');
            } else {
                return redirect()->route('CILs')->with('success', 'Cil Created Successfully');
            }
        } catch (Exception $e) {
            return redirect()->back()->withErrors(['msg' => $e->getMessage()]);
        }
    }

    public function edit($leadId)
    {
        $lead = Lead::findOrFail($leadId);
        // dd($lead);
        $phones = PhoneNumber::where('lead_id', $leadId)->get();
        $users = User::get();
        $countries = Country::all();
        $campaigns = Campaign::all();
        $brokers = Broker::all();

        // Log activity
        $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Leads Edit')
            ->log(auth()->user()->name . ' viewed lead: ' . $lead->name . ' edit page');

        return view('leads.edit', compact('lead', 'users', 'phones', 'countries', 'campaigns', 'brokers'));
    }

    public function update($leadId, LeadRequest $request)
    {
        // Update Lead Data
        $lead = Lead::findOrFail($leadId);

        $lead->name = $request->name;
        $lead->email = $request->email;
        $lead->job_title = $request->job_title;
        $lead->contact_tool = $request->contact_tool ?? null;
        $lead->reply_option = $request->reply_option ?? null;
        $lead->purpose = $request->purpose ?? null;
        $lead->segment = $request->segment ?? null;
        $lead->tag = $request->tag ?? null;
        $lead->note = $request->note ?? null;

        if ($lead->type == 'lead') {
            $lead->campaign_id = $request->campaign_id;
            $lead->source_id = $request->source_id ?? $lead->source_id;
            $lead->came_from = $request->came_from;

            $campaign = Campaign::find($request->campaign_id);
            if (isset($campaign)) {
                $lead->project_id = $campaign->project?->id ?? NULL;
            }
        } else {
            $lead->civil_id = $request->civil_id;
            $lead->broker_id = $request->broker_id;
        }

        $lead->update();

        // Log Activity
        $model = new Lead();
        $user = auth()->user();

        activity()
            ->performedOn($model)
            ->causedBy($user)
            ->event('Update Lead')
            ->log(auth()->user()->name . ' updated lead: ' . $lead->name);

        return redirect()->route('leads.show', $lead->id)->with('success', 'Lead Updated Successfully');
    }

    public function show($leadId, LeadService $leadService)
    {
        // pagination for $data['communications']        
        $data = $leadService->getLeadShowData($leadId);
        dd(count($data));
        $leadService->logActivity('Visit - Leads Profile', auth()->user()->name . ' viewed lead: ' . $data['lead']->name . ' profile');

        return view('leads.show', $data);
    }

    public function destroy($leadId, LeadService $leadService)
    {
        $lead = Lead::findOrFail($leadId);
        
        $leadService->logActivity('Leads - Archive', auth()->user()->name . ' Archived lead: ' . $lead->name);
        $leadService->deleteLeadPhoneNumbers($lead);
        $lead->delete();

        return redirect()->route('leads.index')->with('success', 'Lead Archived Successfully');
    }

    public function importView()
    {
        return view('leads.import');
    }

    public function import(UploadLeadRequest $request)
    {
        if ($request->file) {
            $import = new LeadsImport;

            try {
                Excel::import($import, $request->file);

                $invalidCountryCodes = $import->getInvalidCountryCodes();

                $message = 'Leads uploaded successfully.';
                if (!empty($invalidCountryCodes)) {
                    $message .= ' However, the following country codes were not found: ' . implode(', ', $invalidCountryCodes);
                }

                $model = new Lead();
                $CurrentUser = auth()->user();
                activity()
                    ->performedOn($model)
                    ->causedBy($CurrentUser)
                    ->event('Leads - Import')
                    ->log(auth()->user()->name . ' Imported Leads Sheet');

                return redirect()->route('leads.index')->with('success', $message);
            } catch (Exception $e) {
                return redirect()->back()->with('error', 'An unexpected error occurred during import: ' . $e->getMessage());
            }
        } else {
            return redirect()->back()->with('error', 'Choose a file to upload.');
        }
    }


    public function archive()
    {
        $leads = Lead::onlyTrashed()->get();

        // Log activity
        $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Visit - Leads archive')
            ->log(auth()->user()->name . ' viewed leads archive');

        return view('leads.archive', compact('leads'));
    }

    public function restore($leadId)
    {
        $lead = Lead::withTrashed()->findOrFail($leadId);
        $lead->restore();

        $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Leads - Restore')
            ->log(auth()->user()->name . ' restored lead: ' . $lead->name);

        return redirect()->back()->with('success', 'Lead Restored Successfully');
    }

    public function restoreAll()
    {
        Lead::where(['type' => 'cil'])->onlyTrashed()->restore();

        $leads = new Lead();
        $user = auth()->user();

        activity()
            ->performedOn($leads)
            ->causedBy($user)
            ->event('Restore All buildings')
            ->log('All building Restored by ' . $user->name);

        return redirect()->back();
    }

    public function deletePermanently($id)
    {
        $lead = Lead::withTrashed()->find($id);
        $lead->forceDelete();

        return redirect()->route('leads.index')->with('success', 'Lead Trashed successfully');
    }

    public function bulkAction(Request $request, $leads)
    {

        // try{
        $leads = Lead::whereIn('id', explode(',', $leads))->get();
        $model = new Lead();
        $CurrentUser = auth()->user();

        activity()
            ->performedOn($model)
            ->causedBy($CurrentUser)
            ->event('Leads Bulk-Action')
            ->log(auth()->user()->name . ' made a change into group of leads ');
        foreach ($leads as $lead) {

            if ($request->user_id) {
                $user = User::find($request->user_id);
                $lead->user_id = $request->user_id;
                $lead->status_id = 1;
                $lead->is_delay = 0;
                $lead->save();

                // Calculate Deadline
                $ExtraTime = Setting::where('key', 'ExtraTime')->first();
                $now = Carbon::now();
                $deadline = Carbon::parse($now)->addhours($ExtraTime->value);

                // Store into lead_user table -> we removed the many to many relation
                // we using it to store and manage the deadline of the user and withdrow leads
                $LeadUser = new LeadUser();
                $LeadUser->user_id = $request->user_id;
                $LeadUser->lead_id = $lead->id;
                $LeadUser->status = 'bending';
                $LeadUser->extraTime = $ExtraTime->value;
                $LeadUser->deadline = $deadline;
                $LeadUser->save();

                // Log Activity
                activity()
                    ->performedOn($lead)
                    ->causedBy($CurrentUser)
                    ->event('Leads Assign-user')
                    ->withProperties([
                        'action_creator_name' => $CurrentUser->name,
                        'action_creator_id' => $CurrentUser->id,
                        'action_type' => 'Re-Assign User to Lead',
                        'lead_name' => $lead->name,
                        'lead_id' => $lead->id,
                        'lead_status' => $lead->LeadStatus?->name ?? '',
                        'sales_id' => $user->name,
                        'sales_name' => $user->id,
                    ])
                    ->log($CurrentUser->name . 'Assigned Lead (' . $lead->name . ') to user (' . $user->name . ')');
            }
        }
        return redirect()->route('leads.index')->with('success', 'Lead Updated Successfully');
    }

    public function reAssignUser(Request $request, $leadId)
    {
        // Assign User
        $user = User::find($request->user_id);
        $lead = Lead::find($leadId);
        $lead->user_id = $user->id;
        $lead->status_id = 1;
        $lead->is_delay = 0;
        $lead->save();

        // Calculate Deadline
        $now = Carbon::now();
        $ExtraTime = Setting::where('key', 'ExtraTime')->first();
        $deadline = Carbon::parse($now)->addhours($ExtraTime->value);

        // Store into lead_user table -> we removed the many to many relation
        // we using it to store and manage the deadline of the user and withdrow leads
        $LeadUser = new LeadUser();
        $LeadUser->user_id = $request->user_id;
        $LeadUser->lead_id = $lead->id;
        $LeadUser->status = 'bending';
        $LeadUser->extraTime = $ExtraTime->value;
        $LeadUser->deadline = $deadline;
        $LeadUser->save();

        // Log Activity
        // $model = new Lead();
        $CurrentUser = auth()->user();
        activity()
            ->performedOn($lead)
            ->causedBy($CurrentUser)
            ->event('Leads Assign-user')
            ->withProperties([
                'action_creator_name' => $CurrentUser->name,
                'action_creator_id' => $CurrentUser->id,
                'action_type' => 'Re-Assign User to Lead',
                'lead_name' => $lead->name,
                'lead_id' => $lead->id,
                'lead_status' => $lead->LeadStatus->name,
                'sales_name' => $user->name,
                'sales_id' => $user->id,
            ])
            ->log($CurrentUser->name . 'Assigned Lead (' . $lead->name . ') to user (' . $user->name . ')');

        return redirect()->route('leads.show', $leadId)->with('success', 'Lead Updated Successfully');
    }

    public function delayLeads(Request $request, LeadService $leadService)
    {
        $data = $leadService->getDelayLeadsData($request);

        $leadService->logActivity('Visit - Leads Delay', auth()->user()->name . ' viewed leads Delay');

        return view('leads.delayLeads', $data);
    }

    public function PhoneCode(Request $request, LeadService $leadService)
    {
        $code = $leadService->getPhoneCode($request->country_id);
        return json_encode(['code' => $code]);
    }

    public function campaignSources(Request $request, LeadService $leadService)
    {
        $sources = $leadService->getCampaignSources($request->campaign_id);
        return json_encode(['sources' => $sources]);
    }

    public function CILs(Request $request, CilService $cilService)
    {
        $cilService->logCILsActivity();
        
        $leads = $cilService->buildCILsQuery();
        $cilService->applyCILsFilters($leads, $request);
        
        $leads = $leads->paginate(100)->withQueryString();
        
        $filterData = $cilService->getCILsFilterData();
        $UserChildren = $cilService->getUserChildrenForCILs();
        $leadNumbers = $cilService->calculateMaxPhoneNumbers($leads);
        
        $cilService->logLeadsIndexActivity();
        
        $statuses = $filterData['statuses'];
        $users = $filterData['users'];
        $creators = $filterData['creators'];
        $campaigns = $filterData['campaigns'];
        $countries = $filterData['countries'];
        $projects = Project::all();
        $sources = Source::all();
        $statusTypes = \DB::table('statuses')->pluck('type')->unique()->toArray();
        $events = Event::with('communications')->get();
        
        return view('leads.index', compact('leads', 'users', 'campaigns', 'UserChildren', 'statuses', 'creators', 'leadNumbers', 'countries', 'projects', 'sources', 'statusTypes', 'events'));
    }

    public function confirmCIL(Request $request, $id, LeadService $leadService)
    {
        $leadService->logActivity('Confirm - CILs', auth()->user()->name . ' Confirmed CILs ');
        
        $cil = $leadService->confirmCIL($id);
        return redirect()->route('CILs')->with('success', 'Cil confirmed Successfully');
    }
    
    
    public function checkMobileNumber(Request $request, LeadService $leadService)
    {
        $isExists = $leadService->checkMobileNumberExists($request->mobile_number, $request->country_code);

        return response()->json([
            'exists' => $isExists
        ]);
    }

    public function export(Request $request, $status = null)
    {
        $user = auth()->user();
        
        // Status filtering - same logic as index method
        $statusMapping = [
            'fresh' => 1,
            'interested' => 20,
            'fakeNumber' => 21,
            'meeting' => 27,
            'pending' => 28,
            'followUp' => 29,
            'lowBudget' => 30,
            'noRegisteration' => 31,
            'differentLocation' => 32,
            'closed' => 36,
            'callFailed' => 37,
            'noAnswer' => 38,
        ];

        // Add status to request if it exists
        if ($status && isset($statusMapping[$status])) {
            $request->merge(['status_id' => $statusMapping[$status]]);
        }
        
        // Debug: Log the request parameters
        \Log::info('Leads Export Request Parameters', [
            'all_params' => $request->all(),
            'query_string' => $request->getQueryString(),
            'status_parameter' => $status,
            'status_mapped_id' => $status ? ($statusMapping[$status] ?? null) : null,
            'user_id' => $user->id,
            'user_name' => $user->name
        ]);
        
        activity()
            ->event('export')
            ->log($user->name . ' exported leads data' . ($status ? ' with status: ' . $status : ''));

        return Excel::download(new \App\Exports\LeadsExport($request), 'leads_' . ($status ? $status . '_' : '') . date('Y-m-d_H-i-s') . '.xlsx');
    }


}
