<?php

namespace App\Exports;

use App\Models\User;
use App\Models\Role;
use App\Models\Country;
use App\Models\PhoneNumber;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class UsersExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    public function collection()
    {
        $currentUser = auth()->user();
        $users = User::query();
        
        // Apply user access restrictions
        if ($currentUser->type == 'admin' || $currentUser->role->name == 'Operation Specialist') {
            $users->where('is_blocked', 0);
        } else {
            $users->where('is_blocked', 0);
            $userChildren = array();
            $allUsers = $users->get();
            
            foreach ($allUsers as $user) {
                $parents = $user->getParents()->pluck('id')->toArray();
                if (in_array($currentUser->id, $parents)) {
                    $userChildren[] = $user->id;
                }
            }
            
            $users->whereIn('id', $userChildren);
        }

        // Check if any filters are applied
        $hasFilters = false;

        // Apply filters based on request only if they have actual values
        if ($this->request->name && trim($this->request->name) !== '') {
            $users->where('name', 'LIKE', '%' . trim($this->request->name) . '%');
            $hasFilters = true;
        }

        if ($this->request->email && trim($this->request->email) !== '') {
            $users->where('email', 'LIKE', '%' . trim($this->request->email) . '%');
            $hasFilters = true;
        }

        if ($this->request->role_id && !empty($this->request->role_id)) {
            $users->whereIn('role_id', $this->request->role_id);
            $hasFilters = true;
        }

        if ($this->request->assigned_to && !empty($this->request->assigned_to)) {
            $users->whereIn('assigned_to', $this->request->assigned_to);
            $hasFilters = true;
        }

        if ($this->request->country_id && !empty($this->request->country_id)) {
            $users_id = PhoneNumber::where('country_id', $this->request->country_id)
                ->where('user_id', '!=', NULL)
                ->pluck('user_id')
                ->toArray();
            $users->whereIn('id', $users_id);
            $hasFilters = true;
        }
        
        if ($this->request->phone && trim($this->request->phone) !== '') {
            $users_id = PhoneNumber::where('phone', 'LIKE', '%' . trim($this->request->phone) . '%')
                ->where('user_id', '!=', NULL)
                ->pluck('user_id')
                ->toArray();
            $users->whereIn('id', $users_id);
            $hasFilters = true;
        }

        if ($this->request->time1 && $this->request->time2 && 
            trim($this->request->time1) !== '' && trim($this->request->time2) !== '') {
            $users->whereTime('created_at', '>=', $this->request->time1)
                ->whereTime('created_at', '<=', $this->request->time2);
            $hasFilters = true;
        }

        if ($this->request->date1 && $this->request->date2 && 
            trim($this->request->date1) !== '' && trim($this->request->date2) !== '') {
            $users->whereDate('created_at', '>=', $this->request->date1)
                ->whereDate('created_at', '<=', $this->request->date2);
            $hasFilters = true;
        }

        if ($this->request->is_blocked !== null && $this->request->is_blocked !== '') {
            $users->where('is_blocked', $this->request->is_blocked);
            $hasFilters = true;
        }

        return $users->with(['role', 'phones.country', 'parent'])->get();
    }

    public function headings(): array
    {
        return [
            'User ID',
            'Name',
            'Email',
            'Role',
            'Assigned To',
            'Phone Number 1',
            'Phone Number 2',
            'Country',
            'Status',
            'Created Date',
            'Last Login'
        ];
    }

    public function map($user): array
    {
        return [
            $user->id,
            $user->name,
            $user->email,
            $user->role ? $user->role->name : '--',
            $user->parent ? $user->parent->name : '--',
            $user->phones->count() > 0 ? $user->phones[0]->phone : '--',
            $user->phones->count() > 1 ? $user->phones[1]->phone : '--',
            $user->phones->count() > 0 ? $user->phones[0]->country->name : '--',
            $user->is_blocked ? 'Blocked' : 'Active',
            $user->created_at ? $user->created_at->format('Y-m-d H:i:s') : '--',
            $user->last_login_at ? $user->last_login_at->format('Y-m-d H:i:s') : '--'
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
        ];
    }
} 