<?php

namespace App\Exports;

use App\Models\Source;
use App\Models\Project;
use App\Models\SourceType;
use App\Models\Campaign;
use App\Models\Lead;
use App\Models\Client;
use App\Models\Inquiry;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class SourcesExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    public function collection()
    {
        $sources = Source::query();

        // Check if any filters are applied
        $hasFilters = false;

        // Apply filters based on request only if they have actual values
        if ($this->request->source_id && !empty($this->request->source_id)) {
            $sources->whereIn('id', $this->request->source_id);
            $hasFilters = true;
        }

        if ($this->request->project_id && !empty($this->request->project_id)) {
            $sources->whereIn('project_id', $this->request->project_id);
            $hasFilters = true;
        }

        if ($this->request->sourceType_id && !empty($this->request->sourceType_id)) {
            $sources->whereIn('sourceType_id', $this->request->sourceType_id);
            $hasFilters = true;
        }

        if ($this->request->campaign_id && !empty($this->request->campaign_id)) {
            $sources->whereIn('campaign_id', $this->request->campaign_id);
            $hasFilters = true;
        }

        if ($this->request->time1 && $this->request->time2 && 
            trim($this->request->time1) !== '' && trim($this->request->time2) !== '') {
            $sources->whereTime('created_at', '>=', $this->request->time1)
                ->whereTime('created_at', '<=', $this->request->time2);
            $hasFilters = true;
        }

        if ($this->request->date1 && $this->request->date2 && 
            trim($this->request->date1) !== '' && trim($this->request->date2) !== '') {
            $sources->whereDate('created_at', '>=', $this->request->date1)
                ->whereDate('created_at', '<=', $this->request->date2);
            $hasFilters = true;
        }

        if ($this->request->name && trim($this->request->name) !== '') {
            $sources->where('name', 'LIKE', '%' . trim($this->request->name) . '%');
            $hasFilters = true;
        }

        return $sources->with(['project', 'sourceType', 'campaign', 'leads', 'clients', 'inquiries'])->get();
    }

    public function headings(): array
    {
        return [
            'Source ID',
            'Name',
            'Project Name',
            'Source Type',
            'Campaign Name',
            'Total Leads',
            'Total Clients',
            'Total Inquiries',
            'Created Date',
            'Updated Date'
        ];
    }

    public function map($source): array
    {
        $totalLeads = $source->leads->count();
        $totalClients = $source->clients->count();
        $totalInquiries = $source->inquiries->count();

        return [
            $source->id,
            $source->name,
            $source->project ? $source->project->name : '--',
            $source->sourceType ? $source->sourceType->name : '--',
            $source->campaign ? $source->campaign->name : '--',
            $totalLeads,
            $totalClients,
            $totalInquiries,
            $source->created_at ? $source->created_at->format('Y-m-d H:i:s') : '--',
            $source->updated_at ? $source->updated_at->format('Y-m-d H:i:s') : '--'
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
        ];
    }
} 