<?php

namespace App\Exports;

use App\Models\Phase;
use App\Models\Project;
use App\Models\Building;
use App\Models\Unit;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class PhasesExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    public function collection()
    {
        $user = auth()->user();
        $phases = Phase::query();

        // Apply user access restrictions
        if ($user->type == 'admin' || auth()->user()->role->name == 'Operation Specialist') {
            $phases = $phases;
        } else {
            $phases = $phases->whereHas('users', function ($q) use ($user) {
                $q->where('users.id', $user->id);
            });
        }

        // Check if any filters are applied
        $hasFilters = false;

        // Apply filters based on request only if they have actual values
        if ($this->request->phase_id && !empty($this->request->phase_id)) {
            $phases->whereIn('id', $this->request->phase_id);
            $hasFilters = true;
        }

        if ($this->request->project_id && !empty($this->request->project_id)) {
            $phases->whereIn('project_id', $this->request->project_id);
            $hasFilters = true;
        }

        if ($this->request->time1 && $this->request->time2 && 
            trim($this->request->time1) !== '' && trim($this->request->time2) !== '') {
            $phases->whereTime('created_at', '>=', $this->request->time1)
                ->whereTime('created_at', '<=', $this->request->time2);
            $hasFilters = true;
        }

        if ($this->request->date1 && $this->request->date2 && 
            trim($this->request->date1) !== '' && trim($this->request->date2) !== '') {
            $phases->whereDate('created_at', '>=', $this->request->date1)
                ->whereDate('created_at', '<=', $this->request->date2);
            $hasFilters = true;
        }

        if ($this->request->name && trim($this->request->name) !== '') {
            $phases->where('name', 'LIKE', '%' . trim($this->request->name) . '%');
            $hasFilters = true;
        }

        if ($this->request->type && trim($this->request->type) !== '') {
            $phases->where('type', 'LIKE', '%' . trim($this->request->type) . '%');
            $hasFilters = true;
        }

        return $phases->with(['project', 'buildings.units', 'users'])->get();
    }

    public function headings(): array
    {
        return [
            'Phase ID',
            'Name',
            'Type',
            'Project Name',
            'Project Type',
            'Assigned Users',
            'Total Buildings',
            'Total Units',
            'Free Units',
            'Reserved Units',
            'Contracted Units',
            'Delivered Units',
            'Created Date',
            'Updated Date'
        ];
    }

    public function map($phase): array
    {
        $totalBuildings = $phase->buildings->count();
        $totalUnits = $phase->buildings->flatMap->units->count();
        $freeUnits = $phase->buildings->flatMap->units->where('status', 'free')->count();
        $reservedUnits = $phase->buildings->flatMap->units->where('status', 'reserved')->count();
        $contractedUnits = $phase->buildings->flatMap->units->where('status', 'taken')->count();
        $deliveredUnits = $phase->buildings->flatMap->units->where('status', 'delivered')->count();

        $assignedUsers = $phase->users->pluck('name')->implode(', ');
        if (empty($assignedUsers)) {
            $assignedUsers = '--';
        }

        return [
            $phase->id,
            $phase->name,
            $phase->type ?? '--',
            $phase->project ? $phase->project->name : '--',
            $phase->project ? ($phase->project->is_residential ? 'Residential' : 'Commercial') : '--',
            $assignedUsers,
            $totalBuildings,
            $totalUnits,
            $freeUnits,
            $reservedUnits,
            $contractedUnits,
            $deliveredUnits,
            $phase->created_at ? $phase->created_at->format('Y-m-d H:i:s') : '--',
            $phase->updated_at ? $phase->updated_at->format('Y-m-d H:i:s') : '--'
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
        ];
    }
} 