<?php

namespace App\Exports;

use App\Models\Lead;
use App\Models\User;
use App\Models\Status;
use App\Models\Campaign;
use App\Models\Country;
use App\Models\PhoneNumber;
use App\Models\Communication;
use App\Models\Event;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use DateTime;

class LeadsExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    public function collection()
    {
        $currentUser = auth()->user();
        $leads = Lead::query();
        
        // Debug: Log the request parameters
        \Log::info('LeadsExport collection method', [
            'request_all' => $this->request->all(),
            'user_id' => $currentUser->id,
            'user_type' => $currentUser->type
        ]);
        
        // Apply user access restrictions
        if ($currentUser->type == 'admin' || $currentUser->role->name == 'Marketing Team Leader' || $currentUser->role->name == 'Marketing Manager' || $currentUser->role->name == 'Sales Admin' || $currentUser->role->name == 'Operation Specialist') {
            $leads->where(['is_delay' => 0, 'type' => 'lead']);
        } elseif ($currentUser->assigned_to == null) {
            $childUsers = $currentUser->allChildren()->pluck('id')->toArray();
            if (count($childUsers) > 0) {
                $leads = Lead::where(function ($query) use ($currentUser) {
                    $query->where('created_by', $currentUser->id)
                        ->orWhereIn('user_id', $currentUser->allChildren()->pluck('id')->toArray());
                });
            } else {
                $leads = Lead::where('created_by', $currentUser->id);
            }
        } else {
            $users = User::all();
            $UserChildren = array();

            foreach ($users as $userr) {
                $parents = $userr->getParents()->pluck('id')->toArray();
                if (in_array($currentUser->id, $parents)) {
                    array_push($UserChildren, [
                        'id' => $userr->id,
                        'name' => $userr->name,
                    ]);
                }
            }

            $LeadBulk = array();
            if ($currentUser->leads()->count() != 0) {
                foreach ($currentUser->leads as $lead) {
                    array_push($LeadBulk, $lead->id);
                }
            }
            for ($i = 0; $i < count($UserChildren); $i++) {
                $sales = User::find($UserChildren[$i]['id']);
                if (count($sales->leads) == 0) {
                    continue;
                } else {
                    foreach ($sales->leads as $lead) {
                        array_push($LeadBulk, $lead->id);
                    }
                }
            }
            $LeadBulk = array_unique($LeadBulk);
            $leads->where('is_delay', 0);
            $leads->whereIn('id', $LeadBulk);
        }

        // Apply filters based on request
        if ($this->request->name && trim($this->request->name) !== '') {
            $leads->where('name', 'LIKE', '%' . trim($this->request->name) . '%');
        }

        if ($this->request->email && trim($this->request->email) !== '') {
            $leads->where('email', 'LIKE', '%' . trim($this->request->email) . '%');
        }

        if ($this->request->user_id && !empty($this->request->user_id)) {
            $leads->whereIn('user_id', $this->request->user_id);
        }

        if ($this->request->creator_id && !empty($this->request->creator_id)) {
            $leads->whereIn('created_by', $this->request->creator_id);
        }

        if ($this->request->status_id && !empty($this->request->status_id)) {
            $statusIds = is_array($this->request->status_id) ? $this->request->status_id : [$this->request->status_id];
            $leads->whereIn('status_id', $statusIds);
        }

        if ($this->request->statusTypes && !empty($this->request->statusTypes)) {
            $types = Status::whereIn('type', $this->request->statusTypes)->pluck('id')->toArray();
            $leads->whereIn('status_id', $types);
        }

        if ($this->request->country_id && !empty($this->request->country_id)) {
            $leadIds = PhoneNumber::whereIn('country_id', $this->request->country_id)
                ->whereNotNull('lead_id')
                ->pluck('lead_id')
                ->toArray();
            $leads->whereIn('id', $leadIds);
        }

        if ($this->request->phone && trim($this->request->phone) !== '') {
            $leadIds = PhoneNumber::where('phone', 'LIKE', '%' . trim($this->request->phone) . '%')
                ->whereNotNull('lead_id')
                ->pluck('lead_id')
                ->toArray();
            $leads->whereIn('id', $leadIds);
        }

        if ($this->request->campaign_id && !empty($this->request->campaign_id)) {
            $leads->whereIn('campaign_id', $this->request->campaign_id);
        }

        if ($this->request->project_id && !empty($this->request->project_id)) {
            $leads->whereIn('project_id', $this->request->project_id);
        }

        if ($this->request->source_id && !empty($this->request->source_id)) {
            $leads->whereIn('source_id', $this->request->source_id);
        }

        if ($this->request->time1 && $this->request->time2 && 
            trim($this->request->time1) !== '' && trim($this->request->time2) !== '') {
            $leads->whereTime('created_at', '>=', $this->request->time1)
                ->whereTime('created_at', '<=', $this->request->time2);
        }

        if ($this->request->date1 && $this->request->date2 && 
            trim($this->request->date1) !== '' && trim($this->request->date2) !== '') {
            $leads->whereDate('created_at', '>=', $this->request->date1)
                ->whereDate('created_at', '<=', $this->request->date2);
        }

        // Handle tag filter
        if ($this->request->tag && !empty($this->request->tag)) {
            $leads->whereIn('tag', $this->request->tag);
        }

        // Handle start_date and end_date (confirmation date)
        if ($this->request->start_date && $this->request->end_date && 
            trim($this->request->start_date) !== '' && trim($this->request->end_date) !== '') {
            $leads->whereDate('confirmation_date', '>=', $this->request->start_date)
                ->whereDate('confirmation_date', '<=', $this->request->end_date);
        }

        // Handle lastActio_start_date and lastActio_end_date (last action date)
        if ($this->request->lastActio_start_date && $this->request->lastActio_end_date && 
            trim($this->request->lastActio_start_date) !== '' && trim($this->request->lastActio_end_date) !== '') {
            // This would need to be implemented based on your communication/action tracking logic
            // For now, we'll skip this filter as it requires more complex logic
        }

        $result = $leads->with(['user', 'phone_numbers.country', 'communications', 'LeadStatus', 'source.sourceType', 'campaign', 'project', 'created_by_user'])->get();
        
        // Debug: Log the final count
        \Log::info('LeadsExport final result', [
            'total_leads' => $result->count(),
            'filters_applied' => $this->request->all()
        ]);
        
        return $result;
    }

    public function headings(): array
    {
        return [
            'Lead ID',
            'Name',
            'Email',
            'Phone Number 1',
            'Phone Number 2',
            'Assigned To',
            'Last Comment 1',
            'Last Comment 2',
            'Last Comment 3',
            'Last Action Date',
            'Next Action Date',
            'Status',
            'Note',
            'Lead Target',
            'Source Type',
            'Campaign',
            'Project',
            'Tag',
            'Created By',
            'Created Date',
            'Created Time'
        ];
    }

    public function map($lead): array
    {
        $phoneNumbers = [];
        foreach ($lead->phone_numbers as $phoneNumber) {
            $phoneNumbers[] = ($phoneNumber->country->phonecode ?? '') . ' ' . ($phoneNumber->phone ?? '');
        }

        // Get last 3 comments
        $comments = $lead->communications->sortByDesc('created_at')->take(3)
            ->pluck('description')->toArray();
        $paddedComments = array_pad($comments, 3, '--');

        return [
            $lead->id,
            $lead->name,
            $lead->email ?? '--',
            $phoneNumbers[0] ?? '--',
            $phoneNumbers[1] ?? '--',
            $lead->user ? $lead->user->name : '--',
            $paddedComments[0] ?? '--',
            $paddedComments[1] ?? '--',
            $paddedComments[2] ?? '--',
            $lead->communications->last() ? $lead->communications->last()->created_at->format('Y-m-d H:i') : '--',
            $this->getLastEventDate($lead->id),
            $lead->LeadStatus ? $lead->LeadStatus->name : '--',
            $lead->note ?? '--',
            $lead->source ? $lead->source->name : '--',
            $lead->source && $lead->source->sourceType ? $lead->source->sourceType->name : '--',
            $lead->campaign ? $lead->campaign->name : '--',
            $lead->project ? $lead->project->name : '--',
            $lead->tag ?? '--',
            $lead->created_by_user ? $lead->created_by_user->name : '--',
            $lead->created_at ? $lead->created_at->format('Y-m-d') : '--',
            $lead->created_at ? $lead->created_at->format('H:i') : '--'
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
        ];
    }

    protected function getLastEventDate($leadId)
    {
        $communications = Communication::where('lead_id', $leadId)->first();

        if ($communications) {
            $event = Event::where('communication_id', $communications->id)->first();
            if ($event) {
                $lastEventStart = $event->start;
                $dateString = substr($lastEventStart, 0, 10);
                $dateTime = new DateTime($dateString);
                return $dateTime->format('Y-m-d');
            }
        }
        
        return '--';
    }
}
