<?php

namespace App\Exports;

use App\Models\Client;
use App\Models\PhoneNumber;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class ClientsExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    public function collection()
    {
        $user = auth()->user();
        $clients = Client::query();
        $userClients = $user->accessed_clients->pluck('id')->toArray();

        // Apply user access restrictions
        if ($user->type != 'admin' && !$user->role->hasPermission('clients-readAll')) {
            $clients->whereIn('id', $userClients);
        }

        // Check if any filters are applied
        $hasFilters = false;

        // Apply filters based on request only if they have actual values
        if ($this->request->name && trim($this->request->name) !== '' && $this->request->name !== 'null') {
            $clients->where('name', 'LIKE', '%' . trim($this->request->name) . '%');
            $hasFilters = true;
        }

        
        
        if ($this->request->phone && trim($this->request->phone) !== '') {
            $clients_id = PhoneNumber::where('phone', 'LIKE', '%' . trim($this->request->phone) . '%')
                ->where('client_id', '!=', NULL)
                ->pluck('client_id')
                ->toArray();
            $clients->whereIn('id', $clients_id);
            $hasFilters = true;
        }

        if ($this->request->time1 && $this->request->time2 && 
            trim($this->request->time1) !== '' && trim($this->request->time2) !== '') {
            $clients->whereTime('created_at', '>=', $this->request->time1)
                ->whereTime('created_at', '<=', $this->request->time2);
            $hasFilters = true;
        }

        if ($this->request->date1 && $this->request->date2 && 
            trim($this->request->date1) !== '' && trim($this->request->date2) !== '') {
            $clients->whereDate('created_at', '>=', $this->request->date1)
                ->whereDate('created_at', '<=', $this->request->date2);
            $hasFilters = true;
        }

        return $clients->with(['phone_numbers'])->get();
    }

    public function headings(): array
    {
        return [
            'ID',
            'Client Name',
            'Email',
            'Phone Number 1',
            'Phone Number 2',
            'Phone Number 3',
            'Created Date',
            'Created Time',
        ];
    }

    public function map($client): array
    {
        // Get phone numbers
        $phoneNumbers = $client->phone_numbers;
        $phone1 = $phoneNumbers->count() > 0 ? $phoneNumbers[0]->phone : '';
        $phone2 = $phoneNumbers->count() > 1 ? $phoneNumbers[1]->phone : '';
        $phone3 = $phoneNumbers->count() > 2 ? $phoneNumbers[2]->phone : '';

        return [
            $client->id,
            $client->name,
            $client->email,
            $phone1,
            $phone2,
            $phone3,
            $client->created_at->format('Y-m-d'),
            $client->created_at->format('H:i:s'),
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => [
                'font' => [
                    'bold' => true,
                    'color' => ['rgb' => 'FFFFFF'],
                ],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4472C4'],
                ],
            ],
        ];
    }
} 