<?php

namespace App\Exports;

use App\Models\Building;
use App\Models\Phase;
use App\Models\Project;
use App\Models\Unit;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class BuildingsExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    public function collection()
    {
        $user = auth()->user();
        $buildings = Building::query();

        // Apply user access restrictions
        if (auth()->user()->type == 'admin' || auth()->user()->role->name == 'Operation Specialist') {
            $Allbuildings = $buildings;
        } else {
            $buildings->where('is_active', 1);
            $Allbuildings = $buildings->whereHas('phase', function ($query) {
                $query->whereHas('users', function ($q) {
                    $q->where('users.id', auth()->user()->id);
                });
            });
        }

        // Check if any filters are applied
        $hasFilters = false;

        // Apply filters based on request only if they have actual values
        if ($this->request->building_id && !empty($this->request->building_id)) {
            $Allbuildings->whereIn('id', $this->request->building_id);
            $hasFilters = true;
        }

        if ($this->request->phase_id && !empty($this->request->phase_id)) {
            $Allbuildings->whereIn('phase_id', $this->request->phase_id);
            $hasFilters = true;
        }

        if ($this->request->project_id && !empty($this->request->project_id)) {
            $Allbuildings->whereHas('phase.project', function ($query) {
                $query->whereIn('id', $this->request->project_id);
            });
            $hasFilters = true;
        }

        if ($this->request->time1 && $this->request->time2 && 
            trim($this->request->time1) !== '' && trim($this->request->time2) !== '') {
            $Allbuildings->whereTime('created_at', '>=', $this->request->time1)
                ->whereTime('created_at', '<=', $this->request->time2);
            $hasFilters = true;
        }

        if ($this->request->date1 && $this->request->date2 && 
            trim($this->request->date1) !== '' && trim($this->request->date2) !== '') {
            $Allbuildings->whereDate('created_at', '>=', $this->request->date1)
                ->whereDate('created_at', '<=', $this->request->date2);
            $hasFilters = true;
        }

        if ($this->request->name && trim($this->request->name) !== '') {
            $Allbuildings->where('name', 'LIKE', '%' . trim($this->request->name) . '%');
            $hasFilters = true;
        }

        if ($this->request->type && trim($this->request->type) !== '') {
            $Allbuildings->where('type', 'LIKE', '%' . trim($this->request->type) . '%');
            $hasFilters = true;
        }

        if ($this->request->is_active !== null && $this->request->is_active !== '') {
            $Allbuildings->where('is_active', $this->request->is_active);
            $hasFilters = true;
        }

        return $Allbuildings->with(['phase.project', 'units'])->get();
    }

    public function headings(): array
    {
        return [
            'Building ID',
            'Name',
            'Type',
            'Number of Floors',
            'Phase Name',
            'Project Name',
            'Status',
            'Total Units',
            'Free Units',
            'Reserved Units',
            'Contracted Units',
            'Delivered Units',
            'Created Date',
            'Updated Date'
        ];
    }

    public function map($building): array
    {
        $totalUnits = $building->units->count();
        $freeUnits = $building->units->where('status', 'free')->count();
        $reservedUnits = $building->units->where('status', 'reserved')->count();
        $contractedUnits = $building->units->where('status', 'taken')->count();
        $deliveredUnits = $building->units->where('status', 'delivered')->count();

        return [
            $building->id,
            $building->name,
            $building->type,
            $building->number_of_floors,
            $building->phase ? $building->phase->name : '--',
            $building->phase && $building->phase->project ? $building->phase->project->name : '--',
            $building->is_active ? 'Active' : 'Inactive',
            $totalUnits,
            $freeUnits,
            $reservedUnits,
            $contractedUnits,
            $deliveredUnits,
            $building->created_at ? $building->created_at->format('Y-m-d H:i:s') : '--',
            $building->updated_at ? $building->updated_at->format('Y-m-d H:i:s') : '--'
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
        ];
    }
} 