<?php

namespace App\Exports;

use App\Models\Action;
use App\Models\User;
use App\Models\Client;
use App\Models\Unit;
use App\Models\Ticket;
use App\Models\TicketType;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class ActionsExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    public function collection()
    {
        $actions = $this->buildActionQuery();

        // Check if any filters are applied
        $hasFilters = false;

        // Apply filters based on request only if they have actual values
        if ($this->request->client_id && !empty($this->request->client_id)) {
            $actions->whereIn('client_id', $this->request->client_id);
            $hasFilters = true;
        }

        if ($this->request->unit_id && !empty($this->request->unit_id)) {
            $actions->whereIn('unit_id', $this->request->unit_id);
            $hasFilters = true;
        }

        if ($this->request->ticket_type_id && !empty($this->request->ticket_type_id)) {
            $actions->whereIn('ticket_type_id', $this->request->ticket_type_id);
            $hasFilters = true;
        }

        if ($this->request->user_id && !empty($this->request->user_id)) {
            $actions->whereIn('user_id', $this->request->user_id);
            $hasFilters = true;
        }

        if ($this->request->action_creation1 && $this->request->action_creation2 && 
            trim($this->request->action_creation1) !== '' && trim($this->request->action_creation2) !== '') {
            $actions->whereDate('created_at', '>=', $this->request->action_creation1)
                ->whereDate('created_at', '<=', $this->request->action_creation2);
            $hasFilters = true;
        }

        if ($this->request->ticket_creation1 && $this->request->ticket_creation2 && 
            trim($this->request->ticket_creation1) !== '' && trim($this->request->ticket_creation2) !== '') {
            $tickets = Ticket::whereDate('created_at', '>=', $this->request->ticket_creation1)
                ->whereDate('created_at', '<=', $this->request->ticket_creation2)->pluck('id')->toArray();
            $actions->whereIn('ticket_id', $tickets);
            $hasFilters = true;
        }

        if ($this->request->ticketType_id && !empty($this->request->ticketType_id)) {
            $tickets = Ticket::where('ticket_type_id', $this->request->ticketType_id)->pluck('id')->toArray();
            $actions->whereIn('ticket_id', $tickets);
            $hasFilters = true;
        }

        if ($this->request->ticket_status && !empty($this->request->ticket_status)) {
            $tickets = Ticket::where('status', $this->request->ticket_status)->pluck('id')->toArray();
            $actions->whereIn('ticket_id', $tickets);
            $hasFilters = true;
        }

        if ($this->request->action_type && trim($this->request->action_type) !== '') {
            $actions->where('action_type', 'LIKE', '%' . trim($this->request->action_type) . '%');
            $hasFilters = true;
        }

        if ($this->request->details && trim($this->request->details) !== '') {
            $actions->where('details', 'LIKE', '%' . trim($this->request->details) . '%');
            $hasFilters = true;
        }

        return $actions->with([
            'ticket.ticket_type', 'client.phone_numbers', 'unit', 'created_by'
        ])->get();
    }

    private function buildActionQuery()
    {
        if (auth()->user()->type == 'admin' || auth()->user()->role->name == 'Customer Service Manager') {
            return Action::query();
        }

        // Actions Tree for non-admin users
        $actions = Action::query();
        $users = User::all();
        $CurrentUser = auth()->user();
        $UserChildren = array();

        foreach ($users as $userr) {
            $parents = $userr->getParents()->pluck('id')->toArray();
            if (in_array($CurrentUser->id, $parents)) {
                array_push($UserChildren, [
                    'id' => $userr->id,
                    'name' => $userr->name,
                ]);
            }
        }

        $ActionBulk = array();
        if ($CurrentUser->ticket_actions()->count() != 0) {
            foreach ($CurrentUser->ticket_actions as $action) {
                array_push($ActionBulk, $action->id);
            }
        }
        for ($i = 0; $i < count($UserChildren); $i++) {
            $user = User::find($UserChildren[$i]['id']);
            if (count($user->ticket_actions) == 0) {
                continue;
            } else {
                foreach ($user->ticket_actions as $action) {
                    array_push($ActionBulk, $action->id);
                }
            }
        }
       
        return $actions->whereIn('id', $ActionBulk);
    }

    public function headings(): array
    {
        return [
            'Action ID',
            'Ticket ID',
            'Action Type',
            'Details',
            'Client Name',
            'Client Phone Numbers',
            'Unit Name',
            'Project Name',
            'Ticket Type',
            'Ticket Status',
            'Created By',
            'Assigned User',
            'Action Created Date',
            'Ticket Created Date'
        ];
    }

    public function map($action): array
    {
        $phoneNumbers = '';
        if ($action->client && $action->client->phone_numbers) {
            $phoneNumbers = $action->client->phone_numbers->map(function($phone) {
                return $phone->country->phonecode . ' ' . $phone->phone;
            })->implode(', ');
        }

        return [
            $action->id,
            $action->ticket_id,
            $action->action_type ?? '--',
            $action->details ?? '--',
            $action->client ? $action->client->name : '--',
            $phoneNumbers ?: '--',
            $action->unit ? $action->unit->name : '--',
            $action->unit && $action->unit->building && $action->unit->building->phase && $action->unit->building->phase->project 
                ? $action->unit->building->phase->project->name : '--',
            $action->ticket && $action->ticket->ticket_type ? $action->ticket->ticket_type->name : '--',
            $action->ticket ? $action->ticket->status : '--',
            $action->created_by ? $action->created_by->name : '--',
            $action->assigned_user_name ?? '--',
            $action->created_at ? $action->created_at->format('Y-m-d H:i:s') : '--',
            $action->ticket ? $action->ticket->created_at->format('Y-m-d H:i:s') : '--'
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
        ];
    }
} 